%% Program for the analysis of low-loss EELS maps
%
% Author: Christian Zietlow
%
% Paderborn University | Department of Physics | Nanopatterning - Nanoanalysis - Photonic Materials
% czietlow@uni-paderborn.de
%
% Kindly note that the authors disclaim any liability for damages or consequences arising from the use of the provided scripts.
% Users are responsible for ensuring the proper functioning and application of the code.

% ----------------------------------------------------------------------------------------------------------------------------------------------------
% Contains:
% - Andreas Korinek (2025). Gatan Digital Micrograph file reader (https://www.mathworks.com/matlabcentral/fileexchange/45933-gatan-digital-micrograph-file-reader), MATLAB Central File Exchange. Retrieved November 12, 2025.
% -Yun Pu (2025). Waitbar for Parfor (https://www.mathworks.com/matlabcentral/fileexchange/71083-waitbar-for-parfor), MATLAB Central File Exchange. Retrieved November 12, 2025.
%
%
% This software is provided under the CC-BY 4 License
%
%
%
% You are free to:
% Share — copy and redistribute the material in any medium or format for any purpose, even commercially.
% Adapt — remix, transform, and build upon the material for any purpose, even commercially.
% The licensor cannot revoke these freedoms as long as you follow the license terms.

% Under the following terms:
% Attribution — You must give appropriate credit , provide a link to the license, and indicate if changes were made . You may do so in any reasonable manner, but not in any way that suggests the licensor endorses you or your use.
% No additional restrictions — You may not apply legal terms or technological measures that legally restrict others from doing anything the license permits.
%
% Notices:
% You do not have to comply with the license for elements of the material in the public domain or where your use is permitted by an applicable exception or limitation .
%
% No warranties are given. The license may not give you all of the permissions necessary for your intended use. For example, other rights such as publicity, privacy, or moral rights may limit how you use the material.
%
%-------------------------------------------------------------------------------------------------------------------------------------------------------
%% Programm
function varargout = EELS_Analyzer(varargin)
% EELS_ANALYZER MATLAB code for EELS_Analyzer.fig
%      EELS_ANALYZER, by itself, creates a new EELS_ANALYZER or raises the existing
%      singleton*.
%
%      H = EELS_ANALYZER returns the handle to a new EELS_ANALYZER or the handle to
%      the existing singleton*.
%
%      EELS_ANALYZER('CALLBACK',hObject,eventData,handles,...) calls the local

%      function named CALLBACK in EELS_ANALYZER.M with the given input arguments.
%
%      EELS_ANALYZER('Property','Value',...) creates a new EELS_ANALYZER or raises the
%      existing singleton*.  Starting from the left, property value pairs are
%      applied to the GUI before EELS_Analyzer_OpeningFcn gets called.  An
%      unrecognized property name or invalid value makes property application
%      stop.  All inputs are passed to EELS_Analyzer_OpeningFcn via varargin.
%
%      *See GUI Options on GUIDE's Tools menu.  Choose "GUI allows only one
%      instance to run (singleton)".
%
% See also: GUIDE, GUIDATA, GUIHANDLES

% Edit the above text to modify the response to help EELS_Analyzer

% Last Modified by GUIDE v2.5 13-Aug-2024 10:02:29

% Begin initialization code - DO NOT EDIT
gui_Singleton = 1;
gui_State = struct('gui_Name',       mfilename, ...
    'gui_Singleton',  gui_Singleton, ...
    'gui_OpeningFcn', @EELS_Analyzer_OpeningFcn, ...
    'gui_OutputFcn',  @EELS_Analyzer_OutputFcn, ...
    'gui_LayoutFcn',  [] , ...
    'gui_Callback',   []);
if nargin && ischar(varargin{1})
    gui_State.gui_Callback = str2func(varargin{1});
end

if nargout
    [varargout{1:nargout}] = gui_mainfcn(gui_State, varargin{:});
else
    gui_mainfcn(gui_State, varargin{:});
end
% End initialization code - DO NOT EDIT


% --- Executes just before EELS_Analyzer is made visible.
function EELS_Analyzer_OpeningFcn(hObject, ~, handles, varargin)
% This function has no output args, see OutputFcn.
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
% varargin   command line arguments to EELS_Analyzer (see VARARGIN)
% Choose default command line output for EELS_Analyzer
handles.output = hObject;
% Update handles structure
guidata(hObject, handles);
addpath(genpath('./Scripts'));
handles.sliderListener = addlistener(handles.Slider,'ContinuousValueChange', ...
    @(hObject, event) Slider_Callback(...
    hObject, event, handles));

handles.AxMaxListener = addlistener(handles.CAxisMax,'ContinuousValueChange', ...
    @(hObject, event) CAxisMax_Callback(...
    hObject, event, handles));

handles.AxMinListener = addlistener(handles.CAxisMin,'ContinuousValueChange', ...
    @(hObject, event) CAxisMin_Callback(...
    hObject, event, handles));

if gpuDeviceCount>0
    set(handles.GPU,'Value',1);
end
set(handles.LoadPath,'String',[]);
clear global

% UIWAIT makes EELS_Analyzer wait for user response (see UIRESUME)
% uiwait(handles.figure1);

% --- Outputs from this function are returned to the command line.


function varargout = EELS_Analyzer_OutputFcn(~, ~, handles)
% varargout  cell array for returning output args (see VARARGOUT);
% hObject    handle to figure
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Get default command line output from handles structure
varargout{1} = handles.output;

% --- Executes on button press in Load.
function Load_Callback(~ , ~, handles)
% hObject    handle to Load (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

global tracer

if isempty(tracer)
    tracer=1;
else
    clearvars all
    return
end
pause(0.1)
%% Load Data
[FileName,PathName] = uigetfile({'*.dm4';'*.mat'},'Select File!');
if isequal(FileName,0)
    clear global
    return
else
    [~,~,ext] = fileparts(fullfile(PathName,FileName));
end

 try
    switch ext
        case '.dm4'
            WaitTics        = 5;
            WaitLoad        = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
            [tags, image]   = dmread(fullfile(PathName, FileName));
            FileName  = FileName(1:end-4);
            PathName  = PathName(1:end-1);
            
            image     = double(image);
            ScaleY    = double(tags.ImageList.Unnamed1.ImageData.Calibrations.Dimension.Unnamed0.Scale.Value);
            ScaleX    = double(tags.ImageList.Unnamed1.ImageData.Calibrations.Dimension.Unnamed1.Scale.Value);
            EnergyRes = double(tags.ImageList.Unnamed1.ImageData.Calibrations.Dimension.Unnamed2.Scale.Value);
            
            Exposure  = double(tags.ImageList.Unnamed1.ImageTags.SI.Acquisition.PixelTime0x28s0x29.Value);
            SpecBin   = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.hbin.Value);
            VertBin   = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.vbin.Value);
            Width     = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.width.Value) - 1;
            Top       = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.top.Value)   + 1;
            Height    = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.height.Value)- 1;
            Left      = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.left.Value)  + 1;
            FramePara = [Top,Top+Height,Left,Left+Width];
            
            Dark      = double(tags.ImageList.Unnamed1.ImageTags.EELS.Acquisition.HQDarkCorrection.OriginalDarkCorrection.Value);
            HQAppl    = double(tags.ImageList.Unnamed1.ImageTags.EELS.Acquisition.HQDarkCorrection.Applied.Value);
            try
                HQDark    = double(tags.ImageList.Unnamed1.ImageTags.EELS.Acquisition.HQDarkCorrection.HQDarkCorrection.Value);
                Frames    = double(tags.ImageList.Unnamed1.ImageTags.EELS.Acquisition.HQDarkCorrection.FramesAveraged.Value);
            catch
                HQDark = [];
                Frames = [];
            end
            
            image      = round(image      );                                    % for some reason 'double' adds numbers after the 8th decimal to the array
            ScaleX     = round(ScaleX   ,8);                                    % round removes these
            ScaleY     = round(ScaleY   ,8);
            EnergyRes  = round(EnergyRes,8);
            HQDark     = round(HQDark     );
            Dark       = round(Dark       );
            Frames     = round(Frames     );
            FramePara  = round(FramePara  );
            SpecBin    = round(SpecBin    );
            VertBin    = round(VertBin    );
            
            Processing = char(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.HighLevel.Processing.Value)';
            
            WaitLoad.Send;
            
            %% Calculate
            HQDark    = reshape(HQDark,[1,1,size(HQDark,1)]);
            Dark      = reshape(  Dark,[1,1,size(  Dark,1)]);
            GainRef   = ones([1,1,size(image,3)]);
            
            switch Processing
                case 'Unprocessed'
                otherwise
                    if ~HQAppl && ~isempty(HQDark)
                        hqdark = repmat(HQDark,[size(image,1),size(image,2),1]);
                        dark   = repmat(  Dark,[size(image,1),size(image,2),1]);
                        image  = image + dark - hqdark;
                        clearvars hqdark dark
                    end
            end
            
            Energies  = linspace(0,EnergyRes*(size(image,3)-1),size(image,3));
            FWHM      = zeros(size(image,1),size(image,2));
            ScaleX    = ScaleX*1000;
            ScaleY    = ScaleY*1000;
            Norm      = sum(image,3);
            maxEne    = max(image,[],3);
            minEne    = min(image,[],3);
            ZLP       =(minEne + maxEne) / 2;
            Corr      = zeros([size(image,1),size(image,2)]);
            SizeX     = size(image,2);
            SizeY     = size(image,1);
            SizeZ     = size(image,3);
            Prec      = -round(log10(EnergyRes));
            
            parfor i=1:SizeY
                for j=1:SizeX
                    img             = reshape(image(i,j,:),[1,1,SizeZ]);
                    % Find where the data first drops below half the max.
                    index1          = find(img >= ZLP(i,j), 1, 'first');
                    % Find where the data last rises above half the max.
                    index2          = find(img >= ZLP(i,j), 1, 'last');
                    if index1~=index2
                        FWHM(i,j)       = Energies(1,index2)-Energies(1,index1);
                    else
                        FWHM(i,j)       = NaN;
                    end
                    %% Correct Energies to first Pixel ZLP
                    Corr(i,j)          = Energies(round((index1+index2)/2));
                end
            end
            Energies          = Energies-min(Corr,[],'all');
            Energies          = round(Energies,Prec);
            FWHM(isnan(FWHM)) = round(mean(FWHM(~isnan(FWHM)),'all'),Prec);
            Axis              = [ScaleX,ScaleY];
            Max               = Energies(size(image,3));
            Min               = Energies(1);
            steps             = EnergyRes;
            Steps             = double([steps,2*steps]);
            
            %% Create Logbook
            Logbook = {['Logbook: ', FileName, '  |  Image Type: Org'] ;[ 'ZoomVal: [ 1 , 1 , ',num2str(size(image,2)),' , ',num2str(size(image,2)),' ]' ]; 'PreZLPSig: No' ; ['Back: [ 1 , 1 , ', num2str(size(image,2)),' , ',num2str(size(image,1)),' ]  |  PreZLP: ',num2str(min(Energies)),'eV'];[]};
            WaitLoad.Send;
            
            %% Save FilesDeconv
            DeconvImg    = [];
            DenImg       = [];
            Data         = {[],[],[],[],[]};
            Add          = [];
            DecAdd       = [];
            ZLP          = [];
            ZLP_old      = [];
            ZLPEnergies  = [];
            PreZLP       = [];
            ZLPDrift     = [];
            ZLPFWHM      = [];
            ZLPExp       = [];
            ZLPCut       = [];
            Fine         = [];
            DenLog       = [];
            DenIntLog    = [];
            DeconvLog    = [];
            DecSubLog    = [];
            SubtrZLPLog  = [];
            ZLPLog       = [];
            AlignZLPLog    = [];
           
            LogbookZLP   = {[];[]};
            save(fullfile(tempdir,'ShowOrg.mat'),'image');
            save(fullfile(tempdir,'Original.mat'),'image');
            WaitLoad.Send;
            
            save(fullfile(tempdir,'ZLP1.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
            save(fullfile(tempdir,'ZLP2.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
            save(fullfile(tempdir,'SubtrZLP.mat'),'Data');
            save(fullfile(tempdir,'ShowDen.mat'),'DenImg');
            save(fullfile(tempdir,'ShowDeconv.mat'),'DeconvImg');
            save(fullfile(tempdir,'ShowDecSub.mat'),'DeconvImg');
            save(fullfile(tempdir,'Add.mat'),'Add');
            save(fullfile(tempdir,'DecAdd.mat'),'DecAdd');
            save(fullfile(tempdir,'DenLog.mat'),'DenLog');
            save(fullfile(tempdir,'DenIntLog.mat'),'DenIntLog');
            save(fullfile(tempdir,'DeconvLog.mat'),'DeconvLog');
            save(fullfile(tempdir,'DecSubLog.mat'),'DecSubLog');
            save(fullfile(tempdir,'SubtrZLPLog.mat'),'SubtrZLPLog');
            save(fullfile(tempdir,'ZLP1Log.mat'),'ZLPLog');
            save(fullfile(tempdir,'ZLP2Log.mat'),'ZLPLog');
            save(fullfile(tempdir,'AlignZLP1Log.mat'),'AlignZLPLog');
            save(fullfile(tempdir,'AlignZLP2Log.mat'),'AlignZLPLog');
            save(fullfile(tempdir,'Logbook.txt'),'Logbook');
            save(fullfile(tempdir,'DenZLPLog.txt'),'LogbookZLP');
            save(fullfile(tempdir,'LogbookZLP.txt'),'LogbookZLP');
            WaitLoad.Send;
            
            %% Set Handles
            set(handles.ShowFWHM,'UserData',FWHM);
            set(handles.ShowInt,'UserData',Norm);
            set(handles.ShowDrift,'UserData',[]);
            set(handles.EstimatePhi,'UserData',[]);
            set(handles.ZLPAlign,'UserData',Energies);
            set(handles.Load,'UserData',Axis);
            set(handles.Slider, 'Max', Max, 'Min', Min);
            set(handles.Slider, 'UserData',[Max,Min]);
            set(handles.Slider, 'SliderStep' , Steps );
            set(handles.Slider, 'Value', 0);
            set(handles.CAxisMax, 'Max', 1, 'Min', 0.1);
            set(handles.CAxisMax, 'SliderStep' , [0.01,0.02] );
            set(handles.CAxisMax, 'Value', 1);
            set(handles.CAxisMin, 'Max', 0.9, 'Min', 0);
            set(handles.CAxisMin, 'SliderStep' , [0.01,0.02] );
            set(handles.CAxisMin, 'Value', 0);
            set(handles.MaxEnergy ,'String',num2str(Max));
            set(handles.MinEnergy,'String',num2str(Min));
            set(handles.DispRange,'String',num2str(EnergyRes));
            set(handles.Zoom,'UserData',[1,1,size(image,2),size(image,1)]);
            set(handles.Background,'UserData',{[],[]});
            set(handles.Spec,'UserData',[]);
            set(handles.SingleSpec,'UserData',[]);
            set(handles.LoadPath,'String',fullfile(PathName,FileName));
            set(handles.FileName,'String',FileName);
            set(handles.SavePath,'String',PathName);
            set(handles.AlignPanel,'UserData',[0,0,0]);
            set(handles.DecMeth,'String',{'Fowler';'Gauss';'Weibull';'Background'});
            set(handles.DecMeth,'Value',1);
            set(handles.SubMeth,'String',{'Fowler';'Gauss';'Weibull'});
            set(handles.SubMeth,'Value',1)
            set(handles.PeakNo,'Value',1);
            set(handles.Logbook,'UserData',Logbook);
            set(handles.LoadPeak,'UserData',{[],[];[],[]});
            set(handles.DiffMult,'UserData',{[],[];[],[];[],[]});
            set(handles.PhiVal,'String',num2str(0));
            set(handles.ExposureTime,'String',num2str(Exposure));
            set(handles.BGArtifacts,'UserData',{Dark;HQDark;Energies});
            set(handles.DFandGain,'UserData',Frames);
            set(handles.CorrGainRef,'UserData',GainRef);
            set(handles.Load2DRef,'UserData',{[],[]});
            set(handles.VertBinning,'String',num2str(VertBin));
            set(handles.SpecBinning,'String',num2str(SpecBin));
            set(handles.FramePara,'UserData',FramePara);
            set(handles.Processing,'String',Processing);
            set(handles.CorrSat,'UserData',[]);
            set(handles.CorrSat,'String','Correct Saturation');
            set(handles.text71,'UserData',{[],[]});
            set(handles.BetaConv,'String',1);
            set(handles.BetaCorr,'String',1);
            set(handles.PSF,'UserData',{[],[],[],[]});
            set(handles.PSF,'String','Load EELS PSF');
            
            switch Processing
                case 'Gain Normalized'
                    set(handles.CorrGainRef,'String','Replace Gain Ref');
                otherwise
                    set(handles.CorrGainRef,'String','Apply Gain Ref');
            end
            switch Processing
                case 'Unprocessend'
                    set(handles.None,'Value',1);
                    set(handles.DarkSub,'UserData',3);
                otherwise
                    if isempty(HQDark)
                        set(handles.Dark,'Value',1);
                        set(handles.DarkSub,'UserData',2);
                    else
                        set(handles.HQDark,'Value',1);
                        set(handles.DarkSub,'UserData',1);
                    end
            end
            set(handles.ZLP2D,'UserData',[])
            
            %% Visible ons
            set(handles.text,'Visible','on');
            set(handles.text4,'Visible','on');
            set(handles.text6,'Visible','on');
            set(handles.text7,'Visible','on');
            set(handles.text12,'Visible','on');
            set(handles.ImageFilter,'Visible','on');
            set(handles.SavePanel,'Visible','on');
            set(handles.MinEnergy,'Visible','on');
            set(handles.MaxEnergy,'Visible','on');
            set(handles.DispRange,'Visible','on');
            set(handles.Axis,'Visible','on');
            set(handles.Slider,'Visible','on');
            set(handles.CAxisMax,'Visible','on');
            set(handles.CAxisMin,'Visible','on');
            set(handles.DispEnergy,'Visible','on');
            set(handles.text10,'Visible','on');
            set(handles.ShowFWHM,'Visible','on');
            set(handles.ShowInt,'Visible','on');
            set(handles.Spec,'Visible','on');
            set(handles.SingleSpec,'Visible','on');
            set(handles.ZLPAlign,'Visible','on');
            set(handles.text40,'Visible','on');
            set(handles.text41,'Visible','on');
            set(handles.AlignPanel,'Visible','on');
            set(handles.DispPanel,'Visible','on');
            set(handles.ExposureTime,'Visible','on');
            set(handles.text71,'Visible','on');
            set(handles.text74,'Visible','on');
            set(handles.DFandGain,'Visible','on');
            set(handles.VertBinning,'Visible','on');
            set(handles.SpecBinning,'Visible','on');
            set(handles.FramePara,'Visible','on');
            set(handles.text92,'Visible','on');
            set(handles.Processing,'Visible','on');
            set(handles.text95,'Visible','on');
            
            if isempty(HQDark)
                set(handles.HQDark,'Visible','off')
            else
                set(handles.HQDark,'Visible','on')
            end
            
            %% Visible offs
            set(handles.NoiseBox,'Visible','off');
            set(handles.FitPanel,'Visible','off');         
            set(handles.PeakNo,'Visible','off');
            set(handles.ShowDen,'Visible','off');
            set(handles.ShowDrift,'Visible','off');
            set(handles.ShowOrginal,'Visible','off');
            set(handles.ShowDeconv,'Visible','off');
            set(handles.ShowDecSub,'Visible','off');
            set(handles.ShowPeak,'Visible','off');
            set(handles.Artifacts,'Visible','off');
            set(handles.Background,'Visible','off');
            set(handles.DecIter,'Visible','off');
            set(handles.DecEELS,'Visible','off');
            set(handles.DecMeth,'Visible','off');
            set(handles.DeconvSub,'Visible','off');
            set(handles.DecSubIter,'Visible','off');
            set(handles.AlignPeak,'Visible','off');
            set(handles.ZLPPanel,'Visible','off')
            set(handles.DecFac1,'Visible','off');
            set(handles.SubDecFac1,'Visible','off');
            set(handles.DecFac2,'Visible','off');
            set(handles.SubDecFac2,'Visible','off');
            set(handles.Thres,'Visible','off');
            set(handles.text26,'Visible','off');
            set(handles.text32,'Visible','off');
            set(handles.Logbook,'Visible','off');
            set(handles.LiveFitPanel,'Visible','off');
            set(handles.Delete,'Visible','off');
            set(handles.DeleteFit,'Visible','off');
            set(handles.RestImg,'Visible','off');
            set(handles.text43,'Visible','off');
            set(handles.ShowZLPLog,'Visible','off');
            set(handles.CorrGainRef,'Visible','off');
            set(handles.CorrSat,'Visible','off');
            
            %% Reset Buttonnames
            set(handles.ZLPAlign,'String','Broad Align ZLP');
            
            %% Load Noise Parameters
           try
                load([pwd,'\NoiseParams.para'],'Data', '-mat');
                set(handles.GainVal,'String',Data.Gain);
                set(handles.IntDev,'String',Data.IntDev);
                set(handles.BetaConv,'String',Data.BetaConv);
                set(handles.BetaCorr,'String',Data.BetaCorr);
                set(handles.ReadFull','String',Data.ReadOutFull);
                set(handles.ReadStd,'String',Data.ReadOut);
                set(handles.PSF,'UserData',Data.PSF);
                set(handles.PSF,'String','Reload EELS PSF');
           catch
                set(handles.GainVal,'String',1);
                set(handles.IntDev,'String',0);
                set(handles.BetaConv,'String',1);
                set(handles.BetaCorr,'String',1);
                set(handles.ReadFull','String',0);
                set(handles.ReadStd,'String',0);
                set(handles.PSF,'String','Load EELS PSF');
            end
            
            %% Plot
            UpdatePlot(handles,0);
            WaitLoad.Send;
        case '.mat'
            WaitTics        = 6;
            WaitLoad        = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
            Struct  = load(fullfile(PathName,FileName));
            %% Save Files
            image   = Struct.ShowOrg;
            save(fullfile(tempdir,'ShowOrg.mat'),'image');
            image   = Struct.Original;
            save(fullfile(tempdir,'Original.mat'),'image');
            WaitLoad.Send;
            Data    = Struct.SubtrZLP;
            save(fullfile(tempdir,'SubtrZLP.mat'),'Data');
            DenImg  = Struct.ShowDen;
            WaitLoad.Send;
            save(fullfile(tempdir,'ShowDen.mat'),'DenImg');
            DeconvImg=Struct.ShowDeconv;
            WaitLoad.Send;
            save(fullfile(tempdir,'ShowDeconv.mat'),'DeconvImg');
            WaitLoad.Send;
            DeconvImg= Struct.ShowDecSub;
            save(fullfile(tempdir,'ShowDecSub.mat'),'DeconvImg');
            ZLP         = Struct.ZLP1;
            ZLP_old     = Struct.ZLP1_old;
            ZLPEnergies = Struct.ZLP1Energies;
            ZLPFWHM     = Struct.ZLP1FWHM;
            PreZLP      = Struct.ZLP1PreZLP;
            ZLPDrift    = Struct.ZLP1Drift;
            ZLPExp      = Struct.ZLP1Exp;
            ZLPCut      = Struct.ZLP1Cut;
            Fine        = Struct.ZLP1Fine;
            save(fullfile(tempdir,'ZLP1.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
            ZLP         = Struct.ZLP2;
            ZLP_old     = Struct.ZLP2_old;
            ZLPEnergies = Struct.ZLP2Energies;
            ZLPFWHM     = Struct.ZLP2FWHM;
            PreZLP      = Struct.ZLP2PreZLP;
            ZLPDrift    = Struct.ZLP2Drift;
            ZLPExp      = Struct.ZLP2Exp;
            ZLPCut      = Struct.ZLP2Cut;
            Fine        = Struct.ZLP2Fine;
            save(fullfile(tempdir,'ZLP2.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
            Add = Struct.Add;
            save(fullfile(tempdir,'Add.mat'),'Add');
            DecAdd = Struct.DecAdd;
            save(fullfile(tempdir,'DecAdd.mat'),'DecAdd');
            DenLog       = Struct.DenLog;
            save(fullfile(tempdir,'DenLog.mat'),'DenLog');
            DenIntLog    = Struct.DenIntLog;
            save(fullfile(tempdir,'DenIntLog.mat'),'DenIntLog');
            DeconvLog    = Struct.DeconvLog;
            save(fullfile(tempdir,'DeconvLog.mat'),'DeconvLog');
            DecSubLog    = Struct.DecSubLog;
            save(fullfile(tempdir,'DecSubLog.mat'),'DecSubLog');
            SubtrZLPLog  = Struct.SubtrZLPLog;
            save(fullfile(tempdir,'SubtrZLPLog.mat'),'SubtrZLPLog');
            ZLPLog       = Struct.ZLP1Log;
            save(fullfile(tempdir,'ZLP1Log.mat'),'ZLPLog');
            ZLPLog       = Struct.ZLP2Log;
            save(fullfile(tempdir,'ZLP2Log.mat'),'ZLPLog');
            AlignZLPLog    = Struct.AlignZLP1Log;
            save(fullfile(tempdir,'AlignZLP1Log.mat'),'AlignZLPLog');
            AlignZLPLog    = Struct.AlignZLP2Log;
            save(fullfile(tempdir,'AlignZLP2Log.mat'),'AlignZLPLog');
            
            %% Create Logbooks
            LogbookZLP   = {[];[]};
            Logbook      = {[];[]};
            save(fullfile(tempdir,'Logbook.txt'),'Logbook');
            save(fullfile(tempdir,'DenZLPLog.txt'),'LogbookZLP');
            save(fullfile(tempdir,'LogbookZLP.txt'),'LogbookZLP');
            
            %% Save Handles
            set(handles.LoadPeak,'UserData',Struct.ZLPs);
            set(handles.ShowFWHM,'UserData',Struct.ShowFWHM);
            set(handles.ShowInt,'UserData',Struct.ShowInt);
            set(handles.ShowDrift,'UserData',Struct.ShowDrift);
            set(handles.ZLPAlign,'UserData',Struct.ZLPAlign );
            set(handles.Load,'UserData',Struct.Load);
            set(handles.Slider,'Max',Struct.SliderMax);
            set(handles.Slider,'Min',Struct.SliderMin);
            set(handles.Slider, 'UserData',Struct.SliderMaxMin);
            set(handles.Slider, 'SliderStep',Struct.SliderSteps);
            set(handles.Slider, 'Value',Struct.SliderPos);
            set(handles.CAxisMax,'Max',Struct.CAxisMaxMax);
            set(handles.CAxisMax,'Min',Struct.CAxisMaxMin);
            set(handles.CAxisMax, 'SliderStep',Struct.CAxisMaxSliderStep);
            set(handles.CAxisMax, 'Value',Struct.CAxisMaxPos);
            set(handles.CAxisMin, 'Max',Struct.CAxisMinMax);
            set(handles.CAxisMin, 'Min',Struct.CAxisMinMin);
            set(handles.CAxisMin, 'SliderStep',Struct.CAxisMinSliderStep);
            set(handles.CAxisMin, 'Value',Struct.CAxisMinPos);
            set(handles.MaxEnergy ,'String',Struct.MaxEnergy);
            set(handles.MinEnergy,'String',Struct.MinEnergy);
            set(handles.DispRange,'String',Struct.DispRange);
            set(handles.Zoom,'UserData',Struct.Zoom);
            set(handles.Background,'UserData',Struct.Back);
            set(handles.Spec,'UserData',Struct.Spec);
            set(handles.SingleSpec,'UserData',Struct.SingleSpec);
            set(handles.LoadPath,'String',Struct.LoadPath);
            WaitLoad.Send;
            
            set(handles.DecFac1,'String',Struct.DecFac1);
            set(handles.SubDecFac1,'String',Struct.SubDecFac1);
            set(handles.DecFac2,'String',Struct.DecFac2);
            set(handles.SubDecFac2,'String',Struct.SubDecFac2);
            set(handles.DecSubIter,'String',Struct.DecSubIter);
            set(handles.DecIter,'String',Struct.DecIter);
      
            set(handles.NormImg,'Value',Struct.Norm);
            set(handles.Contours,'Value',Struct.Contours);
            set(handles.ContVal,'String',Struct.ContVal);
            set(handles.RemZLP,'Value',Struct.RemZLP);
            set(handles.DispEnergy,'String',Struct.DispEnergy);
            set(handles.SubMeth,'String',Struct.SubMeth);
            set(handles.SubMeth,'Value',Struct.SubMethPos);
            set(handles.DecMeth,'String',Struct.DecMeth);
            set(handles.DecMeth,'Value',Struct.DecMethPos);
            set(handles.DecEELS,'UserData',Struct.Modus);
            set(handles.SavePath,'String',Struct.SavePath);
            set(handles.FileName,'String',Struct.FileName);
            set(handles.AlignPanel,'UserData',Struct.AlignPanel);
            set(handles.GainVal,'String',Struct.GainVal);
            set(handles.IntDev,'String',Struct.IntDev);
            set(handles.BetaConv,'String',Struct.BetaConv);
            set(handles.BetaCorr,'String',Struct.BetaCorr);
            set(handles.ReadStd,'String',Struct.ReadStd);
            set(handles.ReadFull,'String',Struct.ReadFull);
            set(handles.PhiVal,'String',Struct.PhiVal);
            set(handles.ExposureTime,'String',Struct.ExposureTime);
            set(handles.Thres,'String',Struct.Thres);
            set(handles.DecMeth,'UserData',Struct.Sigma)
            set(handles.Logbook,'UserData',Struct.Log);
            set(handles.DiffMult,'String',Struct.DiffMult);
            set(handles.Width,'String',Struct.Width);
            set(handles.DiffMult,'UserData',Struct.StartPar);
            set(handles.PeakNo,'Value',Struct.PeakNoPos);
            set(handles.EstimatePhi,'UserData',Struct.Cut);
            set(handles.BGArtifacts,'UserData',Struct.BGArtifacts);
            set(handles.DFandGain,'UserData',Struct.DFandGain);
            set(handles.DarkSub,'UserData',Struct.DarkSub);
            set(handles.CorrGainRef,'String',Struct.CorrGainRefString);
            set(handles.CorrGainRef,'UserData',Struct.CorrGainRef);
            set(handles.ZLP2D,'UserData',Struct.ZLP2D);
            set(handles.FramePara,'UserData',Struct.FramePara);
            set(handles.Processing,'String',Struct.Processing);
            set(handles.CorrSat,'UserData',Struct.CorrSat);
            set(handles.CorrSat,'String',Struct.StrCorrSat);
            set(handles.text71,'UserData',Struct.text71);
            set(handles.PSF,'String',Struct.StrPSF);
            set(handles.PSF,'UserData',Struct.PSF);
            
            switch Struct.DarkSub
                case 1
                    set(handles.HQDark,'Value',1);
                case 2
                    set(handles.Dark,'Value',1);
                otherwise
                    set(handles.NoSub,'Value',1);
            end
            
            %% Visibles
            set(handles.text,'Visible',Struct.Vtext);
            set(handles.text4,'Visible',Struct.Vtext4);
            set(handles.text6,'Visible',Struct.Vtext6);
            set(handles.text7,'Visible',Struct.Vtext7);
            set(handles.text12,'Visible',Struct.Vtext12);
            set(handles.SavePanel,'Visible',Struct.VSavePanel);
            set(handles.ImageFilter,'Visible',Struct.VImageFilter);
            set(handles.MinEnergy,'Visible',Struct.VMinEnergy);
            set(handles.MaxEnergy,'Visible',Struct.VMaxEnergy);
            set(handles.DispRange,'Visible',Struct.VDispRange);
            set(handles.Axis,'Visible',Struct.VAxis );
            set(handles.Slider,'Visible',Struct.VSlider );
            set(handles.CAxisMax,'Visible',Struct.VCAxisMax);
            set(handles.CAxisMin,'Visible',Struct.VCAxisMin);
            set(handles.DispEnergy,'Visible',Struct.VDispEnergy);
            set(handles.text10,'Visible',Struct.Vtext10);
            set(handles.ShowFWHM,'Visible',Struct.VFWHM);
            set(handles.ShowInt,'Visible',Struct.VShowInt);
            set(handles.Spec,'Visible',Struct.VSpec);
            set(handles.SingleSpec,'Visible',Struct.VSingleSpec);
            set(handles.ZLPAlign,'Visible',Struct.VZLPAlign);
            set(handles.PeakNo,'Visible',Struct.VPeakNo);
            set(handles.Delete,'Visible',Struct.VDelete);
            set(handles.DeleteFit,'Visible',Struct.VDeleteFit);
            set(handles.VertBinning,'String',Struct.VertBinning);
            set(handles.SpecBinning,'String',Struct.SpecBinning);
            
            set(handles.ShowDen,'Visible',Struct.VShowDen);
            set(handles.ShowDrift,'Visible',Struct.VShowDrift);
            set(handles.ShowOrginal,'Visible',Struct.VShowOrginal);
            set(handles.ShowDeconv,'Visible',Struct.VShowDecvonv);
            set(handles.ShowDecSub,'Visible',Struct.VShowDecSub);
            set(handles.ShowPeak,'Visible',Struct.VShowPeak);
            set(handles.ShowZLPLog,'Visible',Struct.VShowZLPLog);
            set(handles.Artifacts,'Visible',Struct.VArtifacts);
            set(handles.Background,'Visible',Struct.VBack);
            set(handles.DecIter,'Visible',Struct.VDecIter);
            set(handles.DecEELS,'Visible',Struct.VDecEELS);
            set(handles.DeconvSub,'Visible',Struct.VDeconvSub);
            set(handles.AlignPeak,'Visible',Struct.VAlignPeak);
            set(handles.DecSubIter,'Visible',Struct.VDecSubIter);
            
            set(handles.ZLPPanel,'Visible',Struct.VZLPPanel);
            set(handles.DecFac1,'Visible',Struct.VDecFac1);
            set(handles.SubDecFac1,'Visible',Struct.VSubDecFac1);
            set(handles.DecFac2,'Visible',Struct.VDecFac2);
            set(handles.SubDecFac2,'Visible',Struct.VSubDecFac2);
            set(handles.DecMeth,'Visible',Struct.VDecMeth);
            set(handles.Thres,'Visible',Struct.VThres);
            set(handles.text26,'Visible',Struct.Vtext26);
            set(handles.text32,'Visible',Struct.Vtext32);
            set(handles.Logbook,'Visible',Struct.VLog);
            set(handles.LiveFitPanel,'Visible',Struct.VLiveFitPanel);
            set(handles.FitPanel,'Visible',Struct.VFitPanel);
            set(handles.NoiseBox,'Visible',Struct.VNoiseBox);
            set(handles.AlignPanel,'Visible',Struct.VAlignPanel);
            set(handles.DispPanel,'Visible',Struct.VDispPanel);
            set(handles.RestImg,'Visible',Struct.VRestImg);
            set(handles.text40,'Visible',Struct.Vtext40);
            set(handles.text41,'Visible',Struct.Vtext41);
            set(handles.text43,'Visible',Struct.Vtext43);
            set(handles.ExposureTime,'Visible',Struct.VExposureTime);
            set(handles.VertBinning,'Visible',Struct.VVertBinning);
            set(handles.SpecBinning,'Visible',Struct.VSpecBinning);
            set(handles.FramePara,'Visible',Struct.VFramePara);
            set(handles.text92,'Visible',Struct.Vtext92);
            set(handles.text71,'Visible',Struct.Vtext71);
            set(handles.text74,'Visible',Struct.Vtext74);
            set(handles.DFandGain,'Visible',Struct.VDFandGain);
            set(handles.CorrGainRef,'Visible',Struct.VCorrGainRef);
            set(handles.Processing,'Visible',Struct.VProcessing);
            set(handles.text95,'Visible',Struct.Vtext95);
            set(handles.CorrSat,'Visible',Struct.VCorrSat);
            
            if isempty(Struct.DFandGain)
                set(handles.HQDark,'Visible','off')
            else
                set(handles.HQDark,'Visible','on')
            end
            
            %% Names
            set(handles.ZLPAlign,'String',Struct.AlignName);
            WaitLoad.Send;
            UpdatePlot(handles);
            
    end
 catch
     WaitLoad.Destroy;
     warndlg('Wrong data format!')
     clear global
     return
 end
WaitLoad.Destroy;
pause(0.1);
clearvars all
clear global

% --- Executes on button press in DenInt.
function DenInt_Callback(~, ~, handles)
% hObject    handle to DenInt (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

[DenInt,DenImg,DeconvImg,DecAdd,DeconvImgSub] = DenoiseInt(handles);
if isempty(DenInt)
    clearvars all
    return
end
%% Set Handles
save(fullfile(tempdir,'ShowDen.mat'),'DenImg');

if ~isempty(DeconvImg)
    save(fullfile(tempdir,'ShowDeconv.mat'),'DeconvImg');
    save(fullfile(tempdir,'DecAdd.mat'),'DecAdd');
end
if ~isempty(DeconvImgSub)
    DeconvImg = DeconvImgSub;
    save(fullfile(tempdir,'ShowDecSub.mat'),'DeconvImg');
end
set(handles.ShowInt,'UserData',DenInt);
set(handles.ShowDen,'Visible','on');
set(handles.ShowOrginal,'Visible','on');
UpdatePlot(handles,1);

clearvars all
clear global

function DecEELS_Callback(hObject, ~, handles)
% hObject    handle to DecEELS (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

[DeconvImg,DecAdd,Sigma,DecSub] = DeconvEELS(hObject,handles);
if isempty(DeconvImg)
    clearvars all
    return
end
%% Set Handles
save(fullfile(tempdir,'ShowDeconv.mat'),'DeconvImg');
save(fullfile(tempdir,'DecAdd.mat'),'DecAdd');
set(handles.ShowDeconv,'Visible','on');
set(handles.ShowOrginal,'Visible','on');
set(handles.DecMeth,'UserData',Sigma)
if DecSub
    set(handles.DeconvSub,'Visible','on');
    set(handles.DecSubIter,'Visible','on');
    set(handles.SubDecFac1,'Visible','on');
    set(handles.SubDecFac2,'Visible','on');
    set(handles.text43,'Visible','on');
else
    set(handles.DeconvSub,'Visible','off');
    set(handles.DecSubIter,'Visible','off');
    set(handles.SubDecFac1,'Visible','off');
    set(handles.SubDecFac2,'Visible','off');
    set(handles.text43,'Visible','off');
end


UpdatePlot(handles,2);
clearvars all
clear global

% --- Executes on button press in DeconvSub.
function DeconvSub_Callback(~, ~, handles)
% hObject    handle to DeconvSub (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

DeconvImg = DeconvSub(handles) ;
if isempty(DeconvImg)
    clearvars all
    return
end
%% Set Handles

save(fullfile(tempdir,'ShowDecSub.mat'),'DeconvImg');
set(handles.ShowDecSub,'Visible','on');
UpdatePlot(handles,3);
clearvars all
clear global

% --- Executes on button press in Zoom.
function Zoom_Callback(~, ~, handles)
% hObject    handle to Zoom (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

[pos,Logbook] = ZoomImage(handles);
if isempty(pos)
    clearvars all
    return
end
%% Update Logbook
set(handles.Logbook,'UserData',Logbook);

%% Set Handles
set(handles.Zoom,'UserData',pos);

UpdatePlot(handles);
clearvars all
clear global

% --- Executes on button press in ZLPAlign.
function ZLPAlign_Callback(~, ~, handles)
% hObject    handle to ZLPAlign (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%% Get Handles
Fine    = get(handles.AlignPanel,'UserData');
Back    = get(handles.Background,'UserData');
Back    = Back{2};
ZLP     = get(handles.LoadPeak,'UserData');
Modi    = get(handles.SubMeth,'String');
DecModi = get(handles.DecMeth,'String');
Fit     = load(fullfile(tempdir,'SubtrZLP.mat')).Data;
Fit     = Fit{1};

if Fine(1,1)==0
    %% Calculate
    [image,Energies,Corr,Cut1,Cut2,FWHM,MaxMin,MaxDrift,ZLPPos1,ZLPPos2,Logbook]= BroadZLPAlign(handles);
    if isempty(image)
        set(handles.AlignPanel,'UserData',Fine);
        clearvars all
        return
    end
    
    %% Set Handles
    set(handles.LiveFitPanel,'Visible','on');
    set(handles.ZLPAlign,'String','Fine Align ZLP');
    set(handles.AlignPanel,'UserData',[1,ZLPPos1,ZLPPos2]);
    set(handles.ZLPAlign,'UserData',Energies);
    set(handles.ShowDrift,'UserData',Corr);
    save(fullfile(tempdir,'ShowOrg.mat'),'image');
    set(handles.ShowFWHM,'UserData',FWHM);
    set(handles.Slider, 'UserData',MaxMin);
    set(handles.ShowDrift,'Visible','on');
    set(handles.Artifacts,'Visible','on');
    set(handles.Background,'Visible','on');
    set(handles.Thres,'Visible','on');
    set(handles.text26,'Visible','on');
    set(handles.text32,'Visible','on');
    set(handles.Logbook,'Visible','on');
    set(handles.Logbook,'UserData',Logbook);
    set(handles.Background,'UserData',{[],Back});
    UpdateMaxEnergy(handles);
    UpdateMinEnergy(handles);
    Norm = sum(image,3);
    set(handles.ShowInt,'UserData',Norm);
    UpdatePlot(handles);
    set(handles.EstimatePhi,'UserData',[Cut1,Cut2]);
    if ~isempty(Back)
        Add = CreatePadding(handles,image);
        save(fullfile(tempdir,'Add.mat'),'Add');
    end
    set(handles.NoiseBox,'Visible','on');
    msgbox(['BroadAlign corrected max. Drift: ',num2str(MaxDrift),'eV!']);
    
else
    Cut  = get(handles.EstimatePhi,'UserData');
    Cut1 = Cut(1);
    Cut2 = Cut(2);
    if ~isempty(ZLP{1,1}) || ~isempty(ZLP{2,1}) || ~isempty(Fit)
        answer = questdlg('The array lengths change with a realignment. Due to padding, ZLPs have to be denoised again, and fits have to be redone! Continue anyway?','Realignment','Yes','No','No');
        switch answer
            case 'Yes'
            otherwise
                return
        end
    end
    [image,Energies,Corr,Cut1New,Cut2New,MaxMin,Background,DenImg,DeconvImg,ShowDecSub,ZLP_load,MaxDrift,Logbook]= FineZLPAlign(handles);
    if isempty(image)
        set(handles.AlignPanel,'UserData',Fine);
        return
    end
    
    Cut1 = Cut1 + Cut1New;
    Cut2 = Cut2 + Cut2New;
    
    %% Set Handles
    set(handles.EstimatePhi,'UserData',[Cut1,Cut2]);
    set(handles.AlignPanel,'UserData',[2,Fine(1,2:3)]);
    set(handles.ZLPAlign,'UserData',Energies);
    set(handles.ShowDrift,'UserData',Corr);
    set(handles.Background,'UserData',Background);
    save(fullfile(tempdir,'ShowOrg.mat'),'image');
    set(handles.Slider, 'UserData',MaxMin);
    set(handles.LoadPeak,'UserData',ZLP_load);
    set(handles.Logbook,'UserData',Logbook);
    save(fullfile(tempdir,'ShowDen.mat'),'DenImg');
    save(fullfile(tempdir,'ShowDeconv.mat'),'DeconvImg');
    DeconvImg = ShowDecSub;
    save(fullfile(tempdir,'ShowDecSub.mat'),'DeconvImg');
    UpdateMaxEnergy(handles);
    UpdateMinEnergy(handles);
    set(handles.DecIter,'Visible','off');
    set(handles.DecEELS,'Visible','off');
    set(handles.DecFac1,'Visible','off');
    set(handles.DecFac2,'Visible','off');
    set(handles.DecMeth,'Visible','off');
    set(handles.DeconvSub,'Visible','off');
    set(handles.DecSubIter,'Visible','off');
    set(handles.SubDecFac1,'Visible','off');
    set(handles.SubDecFac2,'Visible','off');
    set(handles.LoadPeak,'UserData',{[],[];[],[]});
    set(handles.SubMeth,'String',Modi(1:3,:))
    set(handles.SubMeth,'Value',1);
    set(handles.DecMeth,'String',DecModi(1:4,:))
    set(handles.DecMeth,'Value',1);
    Data = {[],[],[],[],[]};
    save(fullfile(tempdir,'SubtrZLP.mat'),'Data');
    Norm = sum(image,3);
    set(handles.ShowInt,'UserData',Norm);
    UpdatePlot(handles);
    if ~isempty(Back)
        Add = CreatePadding(handles,image);
        save(fullfile(tempdir,'Add.mat'),'Add');
    end
    
    msgbox(['FineAlign corrected max. Drift: ',num2str(MaxDrift),'eV!']);
    
end

clearvars all
clear global

% --- Executes on button press in Save.
function Save_Callback(~, ~, handles)
% hObject    handle to Save (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% --- Executes on button press in DecEELS.

Save(handles);

function DispRange_Callback(~, ~, handles)
% hObject    handle to DispRange (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DispRange as text
%        str2double(get(hObject,'String')) returns contents of DispRange as a double
Val = str2double(get(handles.DispRange,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.DispRange,'String',0)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all
disprange = DispRange(handles);

%% Calculate
set(handles.DispRange,'String',disprange);
MinEnergy_Callback(0,0,handles);
MaxEnergy_Callback(0,0,handles);
UpdatePlot(handles);
clearvars all

% --- Executes on button press in ShowInt.
function ShowInt_Callback(~, ~, handles)
% hObject    handle to ShowInt (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
ShowInt(handles)

% --- Executes on button press in ShowFWHM.
function ShowFWHM_Callback(~, ~, handles)
% hObject    handle to ShowFWHM (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

ShowFWHM(handles);

% --- Executes on button press in ShowDrift.
function ShowDrift_Callback(~, ~, handles)
% hObject    handle to ShowDrift (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

ShowDrift(handles)

% --- Executes on button press in Background.
function Background_Callback(~, ~, handles)
% hObject    handle to Background (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%% Load file
image   = load(fullfile(tempdir,'ShowOrg.mat')).image;

%% Get Handles
Zoom    = get(handles.Zoom,'UserData');
Modus   = get(handles.DecEELS,'UserData');

PSF = get(handles.PSF,'UserData');
if isempty(PSF{1,1})
    msgbox('Upload EELS PSF first!')
else
%% Calculate
    [Data,Logbook] = SelectBack(handles,image);
    if isempty(Data)
       clearvars all
       return
    end

    Add = CreatePadding(handles,image);

%% Set Handles
    set(handles.Zoom,'UserData',Zoom);
    set(handles.Background,'UserData',Data);
    set(handles.Logbook,'UserData',Logbook);
    set(handles.RestImg,'Visible','on');
    set(handles.ZLPPanel,'Visible','on');
    set(handles.FitPanel,'Visible','on');
    save(fullfile(tempdir,'Add.mat'),'Add');

%% Update Pl ot
    UpdatePlot(handles,Modus);
    clear global
end
clearvars all

% --- Executes on button press in SingleSpec.
function SingleSpec_Callback(~, ~, handles)
% hObject    handle to SingleSpec (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


Data      = get(handles.SingleSpec,'UserData');
if ~isempty(Data)
    axes(handles.Axis);
    Coord     = ginput(1);
    Energies  = Data{4};
    No        = Data{3}+1;
    if No==2
        specdiff  = figure;
    elseif No>2
        specdiff  = Data{5};
    end
    SingleSpec      = Data{1};
    if ~isvalid(SingleSpec)
        SingleSpec  = figure;
        specdiff    = [];
        No          = 1;
        Energies    = {};
    end
else
    No              = 1;
    Energies        = {};
    axes(handles.Axis);
    Coord           = ginput(1);
    SingleSpec      = figure;
    specdiff        = [];
end


pos  = round(Coord);
Data = {SingleSpec,pos,No,Energies,specdiff};
set(handles.SingleSpec,'UserData',Data);
UpdateSingleSpec(handles,No);
waitfor(SingleSpec)
Data = {SingleSpec,pos,0,{},[]};
set(handles.SingleSpec,'UserData',Data);
clearvars all


% --- Executes on button press in Spec.
function Spec_Callback(~, ~, handles)
% hObject    handle to Spec (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer1 tracer2

if ~isempty(tracer2)
    clearvars all
    return
else
    tracer1 = 1;
end

%% Get Handles

image = get(handles.CAxisMin,'UserData');
image = image{1};
Data  = get(handles.Spec,'UserData');
Axes  = get(handles.Load,'UserData');

Color=['k','m','c','y','r','b','g'];
if ~isempty(Data)
    
    Energies  = Data{4};
    No        = Data{3}+1;
    if No==2
        specdiff=figure;
    elseif No>2
        specdiff  = Data{5};
    end
    Pos       = Data{2};
    spec      = Data{1};
    if ~isvalid(spec)
        spec     = figure;
        specdiff = [];
        No       = 1;
        Energies = {};
    end
    
    YScale = size(image,1)*(Axes(2)/2);
    XScale = size(image,2)*(Axes(1)/2);
    if Pos(1)> -XScale(1) && Pos(1)+Pos(3)< XScale(end) && Pos(2)> -YScale(1) && Pos(2)+Pos(4)< YScale(end)
        Select        = drawrectangle(handles.Axis,'Position',Pos,'Color',Color(No),'StripeColor','w');
    else
        Select        = drawrectangle(handles.Axis,'Color',Color(No),'StripeColor','w');
    end
    
else
    No=1;
    Energies={};
    Select        = drawrectangle(handles.Axis,'Color',Color(No),'StripeColor','w');
    spec=figure;
    specdiff=[];
end
pos  = Select.Position;

Data = {spec,pos,No,Energies,specdiff};
set(handles.Spec,'UserData',Data);
UpdateSpec(handles,image,No);
while isvalid(spec)
    l = addlistener(Select,'MovingROI',@(src,evt)MoveIt(src,evt,No,handles,image));  % Listens for movement of the ROI
    uiwait;                                                                          % Block program execution
    delete(l);
end
try
    pos = Select.Position;
end
delete(Select);
Data = {spec,pos,0,{},[]};
set(handles.Spec,'UserData',Data);
clearvars all
clear global tracer1

% --- Executes on button press in Artifacts.
function Artifacts_Callback(~, ~, handles)
% hObject    handle to Artifacts (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
%% Get handles
Energies   = get(handles.ZLPAlign,'UserData');
EnergyRes  = get(handles.Slider, 'SliderStep');
EnergyRes  = EnergyRes(1);

%% Calculate
[image,Original,Norm,Logbook,Count] = Artifacts(handles);
if isempty(image)
    clearvars all
    clear global
    return
end

%% Update FWHM

maxEne    = max(image,[],3);
minEne    = min(image,[],3);
ZLP       =(minEne + maxEne) / 2;
SizeX     = size(image,2);
SizeY     = size(image,1);
SizeZ     = size(image,3);
FWHM      = zeros([SizeY,SizeX]);
Prec      = -round(log10(EnergyRes));

parfor i=1:SizeY
    for j=1:SizeX
        img             = reshape(image(i,j,:),[1,1,SizeZ]);
        % Find where the data first drops below half the max.
        index1          = find(img >= ZLP(i,j), 1, 'first');
        % Find where the data last rises above half the max.
        index2          = find(img >= ZLP(i,j), 1, 'last');
        if index1~=index2
            FWHM(i,j)       = Energies(1,index2)-Energies(1,index1);
        else
            FWHM(i,j)       = NaN;
        end
    end
end
FWHM(isnan(FWHM)) = round(mean(FWHM(~isnan(FWHM)),'all'),Prec);

%% Set Handles
set(handles.ShowInt,'UserData',Norm);
set(handles.ShowFWHM,'UserData',FWHM);
set(handles.Logbook,'UserData',Logbook);
save(fullfile(tempdir,'ShowOrg.mat'),'image');

image = Original;
save(fullfile(tempdir,'Original.mat'),'image');
UpdatePlot(handles);
switch Count
    case 1
        msgbox([num2str(Count),' Artifact removed from data!']);
    otherwise
        msgbox([num2str(Count),' Artifacts removed from data!']);
end
clearvars all
clear global


% --- Executes on button press in LoadPeak.
function LoadPeak_Callback(hObject, ~, handles)
% hObject    handle to LoadPeak (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
%% Get handles
No      = get(handles.PeakNo,'Value');

%% Calculata
[ZLP,ZLP_old,ZLPLog,ZLPEnergies,ZLPFWHM,PreZLP,ZLPDrift,ZLPExp,ZLPCut,Fine] = LoadPeak(hObject,handles);
if isempty(ZLP)
    clearvars all
    return
else
    clear global
    Delete_Callback(1, 1, handles)
    global tracer
    tracer=1;
end


%% Set Handles
switch No
    case 1
        set(handles.PeakNo,'Visible','on');
        set(handles.AlignPeak,'Visible','on');
        set(handles.Delete,'Visible','on');
        set(handles.ShowZLPLog,'Visible','on');
        save(fullfile(tempdir,'ZLP1.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
        save(fullfile(tempdir,'ZLP1Log.mat'),'ZLPLog');
        AlignZLPLog = ZLPLog;
        save(fullfile(tempdir,'AlignZLP1Log.mat'),'AlignZLPLog');
    case 2
        set(handles.PeakNo,'Visible','on');
        set(handles.AlignPeak,'Visible','on');
        set(handles.Delete,'Visible','on');
        set(handles.ShowZLPLog,'Visible','on');
        save(fullfile(tempdir,'ZLP2.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
        save(fullfile(tempdir,'ZLP2Log.mat'),'ZLPLog');
        AlignZLPLog = ZLPLog;
        save(fullfile(tempdir,'AlignZLP2Log.mat'),'AlignZLPLog');
end
clearvars all
clear global

function UpdateMaxEnergy(handles)

[Pos,NewMin,NewMax]= MaxEnergy(handles);

%% Set Handles
set(handles.Slider,'Min', NewMin, 'Max', NewMax);
set(handles.MaxEnergy,'String',NewMax);
set(handles.Slider,'Value',Pos);
clearvars all

function UpdateMinEnergy(handles)

[Pos,NewMin,NewMax]= MinEnergy(handles);

%% Set Handles
set(handles.Slider,'Min', NewMin, 'Max', NewMax);
set(handles.MinEnergy,'String',NewMin);
set(handles.Slider,'Value',Pos);
clearvars all

function DispEnergy_Callback(~, ~, handles)
% hObject    handle to DispEnergy (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DispEnergy as text
%        str2double(get(hObject,'String')) returns contents of DispEnergy as a double
Val = str2double(get(handles.DispEnergy,'String'));

if isnan(Val) || ~isreal(Val)
    set(handles.DispEnergy,'String',0)
    warndlg('Input must be numerical and real!')
end

Value = DispEnergy(handles);

%% Set Handles
set(handles.Slider,'Value',Value);
UpdatePlot(handles);
clearvars all

% --- Executes on button press in Delete.
function Delete_Callback(~, ~, handles)
% hObject    handle to Delete (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


global tracer

if isempty(tracer)
    tracer=1;
else
    clearvars all
    return
end
%% Get handles
No        = get(handles.PeakNo,'Value');
Modi      = get(handles.SubMeth,'String');
ModNo     = get(handles.SubMeth,'Value');
DecModi   = get(handles.DecMeth,'String');
DecModiNo = get(handles.DecMeth,'Value');
ZLPs      = get(handles.LoadPeak,'UserData');
ZLP1      = load(fullfile(tempdir,'ZLP1.mat'));
ZLP2      = load(fullfile(tempdir,'ZLP2.mat'));
ZLP1      = ZLP1.ZLP;
ZLP2      = ZLP2.ZLP;
switch No
    case 1
        ZLP         = [];
        ZLP_old     = [];
        ZLPEnergies = [];
        ZLPFWHM     = [];
        PreZLP      = [];
        ZLPDrift    = [];
        ZLPExp      = [];
        ZLPCut      = [];
        Fine        = [];
        save(fullfile(tempdir,'ZLP1.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
        ZLPLog = '';
        save(fullfile(tempdir,'ZLP1Log.mat'),'ZLPLog');
        AlignZLPLog = '';
        save(fullfile(tempdir,'AlignZLP1Log.mat'),'AlignZLPLog');
        
        if ~isempty(ZLPs{2,1})
            set(handles.SubMeth,'String',[Modi(1:3,:);'ZLP 2'])
            if ModNo>4
                set(handles.SubMeth,'Value',4);
            end
            set(handles.DecMeth,'String',[DecModi(1:4,:);'ZLP 2'])
            if DecModiNo>5
                set(handles.DecMeth,'Value',5);
            end
        elseif isempty(ZLPs{2,1}) && ~isempty(ZLP2)
            set(handles.SubMeth,'String',Modi(1:3,:))
            if ModNo>3
                set(handles.SubMeth,'Value',1);
            end
            set(handles.DecMeth,'String',DecModi(1:4,:))
            if DecModiNo>4
                set(handles.DecMeth,'Value',1);
            end
        else
            set(handles.PeakNo,'Visible','off');
            set(handles.PeakNo,'Value',1);
            set(handles.SubMeth,'String',Modi(1:3,:))
            if ModNo>3
                set(handles.SubMeth,'Value',1);
            end
            set(handles.DecMeth,'String',DecModi(1:4,:))
            if DecModiNo>4
                set(handles.DecMeth,'Value',1);
            end
            
        end
        set(handles.LoadPeak,'UserData',{[],[];ZLPs{2,:}});
        
        set(handles.AlignPeak,'Visible','off');
        set(handles.ShowPeak,'Visible','off');
        set(handles.Delete,'Visible','off');
        set(handles.ShowZLPLog,'Visible','off');
        
    case 2
        ZLP         = [];
        ZLP_old     = [];
        ZLPEnergies = [];
        ZLPFWHM     = [];
        PreZLP      = [];
        ZLPDrift    = [];
        ZLPExp      = [];
        ZLPCut      = [];
        Fine        = [];
        save(fullfile(tempdir,'ZLP2.mat'),'ZLP','ZLP_old','ZLPEnergies','ZLPFWHM','PreZLP','ZLPDrift','ZLPExp','ZLPCut','Fine');
        ZLPLog = '';
        save(fullfile(tempdir,'ZLP2Log.mat'),'ZLPLog');
        AlignZLPLog = '';
        save(fullfile(tempdir,'AlignZLP2Log.mat'),'AlignZLPLog');
        
        if ~isempty(ZLPs{1,1})
            set(handles.SubMeth,'String',[Modi(1:3,:);'ZLP 1'])
            if ModNo>4
                set(handles.SubMeth,'Value',4);
            end
            set(handles.DecMeth,'String',[DecModi(1:4,:);'ZLP 1'])
            if DecModiNo>5
                set(handles.DecMeth,'Value',5);
            end
        elseif isempty(ZLPs{1,1}) && ~isempty(ZLP1)
            set(handles.SubMeth,'String',Modi(1:3,:))
            if ModNo>3
                set(handles.SubMeth,'Value',1);
            end
            set(handles.DecMeth,'String',DecModi(1:4,:))
            if DecModiNo>4
                set(handles.DecMeth,'Value',1);
            end
        else
            set(handles.PeakNo,'Visible','off');
            set(handles.PeakNo,'Value',1);
            set(handles.SubMeth,'String',Modi(1:3,:))
            if ModNo>3
                set(handles.SubMeth,'Value',1);
            end
            set(handles.DecMeth,'String',DecModi(1:4,:))
            if DecModiNo>4
                set(handles.DecMeth,'Value',1);
            end
        end
        set(handles.LoadPeak,'UserData',{ZLPs{1,:};[],[]});
        
        set(handles.AlignPeak,'Visible','off');
        set(handles.ShowPeak,'Visible','off');
        set(handles.Delete,'Visible','off');
        set(handles.ShowZLPLog,'Visible','off');
end

clearvars all
clear global

function Slider_Callback(~, ~, handles)
% hObject    handle to Slider (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider
UpdateImage(handles);
clearvars all

function MinEnergy_Callback(~, ~, handles)
% hObject    handle to MinEnergy (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of MinEnergy as text
%        str2double(get(hObject,'String')) returns contents of MinEnergy as a double
Val = str2double(get(handles.MinEnergy,'String'));

if isnan(Val) || ~isreal(Val)
    set(handles.MinEnergy,'String',0)
    warndlg('Input must be numerical and real!')
end

UpdateMinEnergy(handles);
UpdatePlot(handles);
clearvars all

function MaxEnergy_Callback(~, ~, handles)
% hObject    handle to MaxEnergy (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of MaxEnergy as text
%        str2double(get(hObject,'String')) returns contents of MaxEnergy as a double

Val = str2double(get(handles.MaxEnergy,'String'));

if isnan(Val) || ~isreal(Val)
    set(handles.MaxEnergy,'String',5)
    warndlg('Input must be numerical and real!')
end

UpdateMaxEnergy(handles);
UpdatePlot(handles);
clearvars all

function ShowOrginal_Callback(~, ~, handles)
% hObject    handle to ShowOrginal (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
%% Set Handles
UpdatePlot(handles,0);
clearvars all

% --- Executes on button press in ShowDen.
function ShowDen_Callback(~, ~, handles)
% hObject    handle to ShowDen (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
%% Set Handles
UpdatePlot(handles,1);
clearvars all

% --- Executes on button press in ShowDeconv.
function ShowDeconv_Callback(~,~, handles)
% hObject    handle to ShowDeconv (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
%% Set Handles
UpdatePlot(handles,2);
clearvars all

% --- Executes on button press in NormImg.
function NormImg_Callback(~, ~, handles)
% hObject    handle to NormImg (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of NormImg
UpdatePlot(handles);
clearvars all

% --- Executes on button press in RemZLP.
function RemZLP_Callback(~, ~, handles)
% hObject    handle to RemZLP (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of RemZLP
UpdatePlot(handles);
clearvars all

function DecIter_Callback(~, ~,handles)
% hObject    handle to DecIter (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DecIter as text
%        str2double(get(hObject,'String')) returns contents of DecIter as a double
Val = str2double(get(handles.DecIter,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.DecIter,'String',150)
    warndlg('Input must be numerical, real and positive!')
elseif ~isinteger(Val)
    set(handles.DecIter,'String',round(Val))
end
clearvars all

% --- Executes on slider movement.
function CAxisMax_Callback(~, ~, handles)
% hObject    handle to CAxisMax (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of sliderf
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider
PlotImg(handles);
clearvars all

% --- Executes on slider movement.
function CAxisMin_Callback(~, ~, handles)
% hObject    handle to CAxisMax (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'Value') returns position of slider
%        get(hObject,'Min') and get(hObject,'Max') to determine range of slider
PlotImg(handles);
clearvars all

% --- Executes on button press in Contours.
function Contours_Callback(~, ~, handles)
% hObject    handle to Contours (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of Contours
PlotImg(handles);
clearvars all

function FileName_Callback(~, ~, handles)
% hObject    handle to FileName (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of FileName as text
%        str2double(get(hObject,'String')) returns contents of FileName as a doubl
String = get(handles.FileName,'String');

i      = strfind(String,'\');
String(i)=[];
j      = strfind(String,'/');
String(j)=[];
k      = strfind(String,':');
String(k)=[];
l      = strfind(String,'*');
String(l)=[];
m      = strfind(String,'"');
String(m)=[];
n      = strfind(String,'>');
String(n)=[];
o      = strfind(String,'<');
String(o)=[];
p      = strfind(String,'|');
String(p)=[];

if ~isempty(i) || ~isempty(j) ||~isempty(k) ||~isempty(l) || ~isempty(m) || ~isempty(n) || ~isempty(o) || ~isempty(p)
    
    set(handles.FileName,'String',String);
    warndlg('Name must not contain any of the following characters: \ / : * ? " < > |')
end


% --- Executes on button press in Crit.
function Crit_Callback(~, ~, ~)
% hObject    handle to Crit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of Crit

% --- Executes on button press in SubtrZLP.
function SubtrZLP_Callback(hObject, ~, handles)
% hObject    handle to SubtrZLP (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

Data = SubtrZLP(hObject,handles);
if isempty(Data)
    return
end
save(fullfile(tempdir,'SubtrZLP.mat'),'Data');
set(handles.RemZLP,'Visible','on')
set(handles.DecIter,'Visible','on');
set(handles.DecEELS,'Visible','on');
set(handles.DecMeth,'Visible','on');
set(handles.DecFac1,'Visible','on');
set(handles.DecFac2,'Visible','on');
set(handles.DeleteFit,'Visible','on');

UpdatePlot(handles);
clearvars all

function DecSubIter_Callback(~, ~, handles)
% hObject    handle to DecSubIter (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DecSubIter as text
%        str2double(get(hObject,'String')) returns contents of DecSubIter as a double

Val = str2double(get(handles.DecSubIter,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.DecSubIter,'String',150)
    warndlg('Input must be numerical, real and positive!')
elseif ~isinteger(Val)
    set(handles.DecSubIter,'String',round(Val))
end
clearvars all

% --- Executes on button press in ShowDecSub.
function ShowDecSub_Callback(~, ~, handles)
% hObject    handle to ShowDecSub (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
UpdatePlot(handles,3);
clearvars all

% --- Executes on selection change in SubMeth.
function SubMeth_Callback(~, ~, handles)
% hObject    handle to SubMeth (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns SubMeth contents as cell array
%        contents{get(hObject,'Value')} returns selected item from SubMeth
Method    = get(handles.LiveFitPanel,'UserData');
if isempty(Method)
    clearvars all
    return
end

if Method{1}==2
    src      = Method{2};
    fig      = Method{3};
    Spec     = Method{4};
    Energies = Method{5};
    FWHM     = Method{6};
    LiveFitZLP(src,1,fig,handles,Spec,Energies,FWHM);
end
clearvars all



function ContVal_Callback(~,~, handles)
% hObject    handle to ContVal (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of ContVal as text
%        str2double(get(hObject,'String')) returns contents of ContVal as a double
Val = str2double(get(handles.ContVal,'String'));

if isnan(Val) || ~isreal(Val)
    set(handles.ContVal,'String',0.8)
    warndlg('Input must be numerical and real!')
elseif 0 >= Val || Val>= 1
    set(handles.ContVal,'String',0.8)
    warndlg('Input must be between 0 and 1!')
end


UpdatePlot(handles)
clearvars all

% --- Executes on button press in GPU.
function GPU_Callback(~, ~, ~)
% hObject    handle to GPU (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: get(hObject,'Value') returns toggle state of GPU

function IntIter_Callback(~, ~, handles)
% hObject    handle to IntIter (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of IntIter as text
%        str2double(get(hObject,'String')) returns contents of IntIter as a double

Val = str2double(get(handles.IntIter,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.IntIter,'String',150)
    warndlg('Input must be numerical, real and positive!')
elseif ~isinteger(Val)
    set(handles.IntIter,'String',round(Val))
end
clearvars all

function IntFac1_Callback(~, ~, handles)
% hObject    handle to IntFac1 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of IntFac1 as text
%        str2double(get(hObject,'String')) returns contents of IntFac1 as a double

Val = str2double(get(handles.IntFac1,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.IntFac1,'String',1)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all

function DecFac1_Callback(~, ~, handles)
% hObject    handle to DecFac1 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DecFac1 as text
%        str2double(get(hObject,'String')) returns contents of DecFac1 as a double
Val = str2double(get(handles.DecFac1,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.DecFac1,'String',1)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all

function SubDecFac1_Callback(~, ~,handles)
% hObject    handle to SubDecFac1 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of SubDecFac1 as text
%        str2double(get(hObject,'String')) returns contents of SubDecFac1 as a double
Val = str2double(get(handles.SubDecFac1,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.SubDecFac1,'String',1)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all

% --- Executes when user attempts to close figure1.
function figure1_CloseRequestFcn(hObject, ~, ~)
% hObject    handle to figure1 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hint: delete(hObject) closes the figure

try
    delete(fullfile(tempdir,'ShowOrg.mat'));
end
try
    delete(fullfile(tempdir,'Original.mat'));
end
try
    delete(fullfile(tempdir,'SubtrZLP.mat'));
end
try
    delete(fullfile(tempdir,'ShowDen.mat'));
end
try
    delete(fullfile(tempdir,'ShowDeconv.mat'));
end
try
    delete(fullfile(tempdir,'ShowDecSub.mat'));
end
try
    delete(fullfile(tempdir,'Add.mat'));
end
try
    delete(fullfile(tempdir,'DecAdd.mat'));
end
try
    delete(fullfile(tempdir,'DenLog.mat'));
end
try
    delete(fullfile(tempdir,'DenIntLog.mat'));
end
try
    delete(fullfile(tempdir,'DeconvLog.mat'));
end
try
    delete(fullfile(tempdir,'DecSubLog.mat'));
end
try
    delete(fullfile(tempdir,'ZLP1Log.mat'));
end
try
    delete(fullfile(tempdir,'ZLP2Log.mat'));
end
try
    delete(fullfile(tempdir,'ZLP1.mat'));
end
try
    delete(fullfile(tempdir,'ZLP2.mat'));
end
try
    delete(fullfile(tempdir,'AlignZLP1Log.mat'));
end
try
    delete(fullfile(tempdir,'AlignZLP2Log.mat'));
end
try
    delete(fullfile(tempdir,'SubtrZLPLog.mat'));
end
try
    delete(fullfile(tempdir,'Logbook.txt'));
end
try
    delete(fullfile(tempdir,'DenZLPLog.txt'));
end
try
    delete(fullfile(tempdir,'LogbookZLP.txt'));
end

clearvars all
clear global
clear FineZLPAlign SubtrZLP CreateKernel
delete(hObject);

% --- Executes on button press in FullSave.
function FullSave_Callback(~, ~, handles)
% hObject    handle to FullSave (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

FullSave(handles);


function GainVal_Callback(~, ~, handles)
% hObject    handle to GainVal (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of GainVal as text
%        str2double(get(hObject,'String')) returns contents of GainVal as a double

Val  = str2double(get(handles.GainVal,'String'));
Back = get(handles.Background,'UserData');
if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.GainVal,'String',1)
    warndlg('Input must be numerical, real and positive!')
    clearvars all
    return
end

Back    = Back{2};
if ~isempty(Back)
    image   = load(fullfile(tempdir,'ShowOrg.mat')).image;
    Add     = CreatePadding(handles,image);
    save(fullfile(tempdir,'Add.mat'),'Add');
end
clearvars all

% --- Executes on selection change in DecMeth.
function DecMeth_Callback(~, ~, handles)
% hObject    handle to DecMeth (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns DecMeth contents as cell array
%        contents{get(hObject,'Value')} returns selected item from DecMeth
Method     = get(handles.LiveFitPanel,'UserData');
if isempty(Method)
    clearvars all
    return
end
switch Method{1}
    case 3
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        LiveFitKernel(src,1,fig,handles,Spec,Energies,FWHM)
    otherwise
        clearvars all
        return
end




function Thres_Callback(~, ~, handles)
% hObject    handle to Thres (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of Thres as text
%        str2double(get(hObject,'String')) returns contents of Thres as a double
Val = str2double(get(handles.Thres,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.Thres,'String',10)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all


% --- Executes on button press in Logbook.
function Logbook_Callback(~, ~, handles)
% hObject    handle to Logbook (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
%% Get handles
modus         = get(handles.DecEELS,'UserData');
if modus==0
    Logbook   = get(handles.Logbook,'UserData');
    NormImg   = 1;
    RemZLP    = get(handles.RemZLP,'Value');
elseif modus==1
    Logbook   = load(fullfile(tempdir,'DenLog.mat')).DenLog ;
    NormImg   = get(handles.NormImg,'Value');
    RemZLP    = get(handles.RemZLP,'Value');
elseif modus==2
    Logbook   = load(fullfile(tempdir,'DeconvLog.mat')).DeconvLog ;
    NormImg   = get(handles.NormImg,'Value');
    RemZLP    = 0;
elseif modus==3
    Logbook   = load(fullfile(tempdir,'DecSubLog.mat')).DecSubLog ;
    NormImg   = get(handles.NormImg,'Value');
    RemZLP    = 0;
end

if ~NormImg
    LogDenInt      = load(fullfile(tempdir,'DenIntLog.mat')).DenIntLog ;
    Logbook{end+1} = [];
    Logbook{end+1} ='-----------------------------------------------------------------------------';
    Logbook{end+1} = [];
    Logbook{end+1} = 'Denoised Intensities: ';
    Logbook        = [Logbook;LogDenInt];
end

if RemZLP && modus==0
    SubtrZLPLog    = load(fullfile(tempdir,'SubtrZLPLog.mat')).SubtrZLPLog;
    Logbook{end+1} = [];
    Logbook{end+1} = '-----------------------------------------------------------------------------';
    Logbook{end+1} = [];
    Logbook{end+1} = 'ZLP Subtraction: ';
    for i=1:size(SubtrZLPLog,1)
        SubtrZLPLog{i,:} = ['  ',SubtrZLPLog{i,:}];
    end
    Logbook        = [Logbook;SubtrZLPLog];
end

fileID = fopen(fullfile(tempdir,'Logbook.txt'),'wt');
for i=1:size(Logbook,1)
    fprintf(fileID,'%s\n',Logbook{i});
end
fclose(fileID);
winopen(fullfile(tempdir,'Logbook.txt'))

clearvars all

% --- Executes on selection change in FitDispOpt.
function FitDispOpt_Callback(~, ~, handles)
% hObject    handle to FitDispOpt (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns FitDispOpt contents as cell array
%        contents{get(hObject,'Value')} returns selected item from FitDispOpt
Method     = get(handles.LiveFitPanel,'UserData');
if isempty(Method)
    return
end

switch Method{1}
    case 1
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        
        LiveFitAlign(src,1,fig,handles,Spec,Energies,FWHM);
    case 2
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        LiveFitZLP(src,1,fig,handles,Spec,Energies,FWHM);
    case 3
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        LiveFitKernel(src,1,fig,handles,Spec,Energies,FWHM)
end

function DiffMult_Callback(~, ~, handles)
% hObject    handle to DiffMult (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DiffMult as text
%        str2double(get(hObject,'String')) returns contents of DiffMult as a double
Val = str2double(get(handles.DiffMult,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.DiffMult,'String',100)
    warndlg('Input must be numerical, real and positive!')
end

Method     = get(handles.LiveFitPanel,'UserData');
if isempty(Method)
    clearvars all
    return
end

switch Method{1}
    case 1
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        LiveFitAlign(src,1,fig,handles,Spec,Energies,FWHM);
    case 2
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        LiveFitZLP(src,1,fig,handles,Spec,Energies,FWHM);
    case 3
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        LiveFitKernel(src,1,fig,handles,Spec,Energies,FWHM)
end
clearvars all

function Width_Callback(~,~, handles)

% hObject    handle to Width (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of Width as text
%        str2double(get(hObject,'String')) returns contents of Width as a double

Val = str2double(get(handles.Width,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.Width,'String',3)
    warndlg('Input must be numerical, real and positive!')
end

Method     = get(handles.LiveFitPanel,'UserData');
Width      = str2double(get(handles.Width,'String'));
if isempty(Method)
    return
end

switch Method{1}
    case 1
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        pos      = src.Position;
        Limit1   = -mean(FWHM,'all')*Width/2;
        Limit2   =  mean(FWHM,'all')*Width/2;
        EStart   = pos(1);
        EWidth   = pos(3);
        EStart   =  max(EStart,Limit1);
        if EStart+EWidth > Limit2
            EWidth =  Limit2-EStart;
        end
        pos(1) = EStart;
        pos(3) = EWidth;
        src.Position = pos;
        LiveFitAlign(src,1,fig,handles,Spec,Energies,FWHM);
    case 2
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        pos      = src.Position;
        Limit1   = -mean(FWHM,'all')*Width/2;
        Limit2   =  mean(FWHM,'all')*Width/2;
        EStart   = pos(1);
        EWidth   = pos(3);
        EStart   =  max(EStart,Limit1);
        if EStart+EWidth > Limit2
            EWidth =  Limit2-EStart;
        end
        pos(1) = EStart;
        pos(3) = EWidth;
        src.Position = pos;
        LiveFitZLP(src,1,fig,handles,Spec,Energies,FWHM);
    case 3
        src      = Method{2};
        fig      = Method{3};
        Spec     = Method{4};
        Energies = Method{5};
        FWHM     = Method{6};
        pos      = src.Position;
        Limit1   = -mean(FWHM,'all')*Width/2;
        Limit2   =  mean(FWHM,'all')*Width/2;
        EStart   = pos(1);
        EWidth   = pos(3);
        EStart   =  max(EStart,Limit1);
        if EStart+EWidth > Limit2
            EWidth =  Limit2-EStart;
        end
        pos(1) = EStart;
        pos(3) = EWidth;
        src.Position = pos;
        LiveFitKernel(src,1,fig,handles,Spec,Energies,FWHM)
end

function IntFac2_Callback(~, ~, handles)
% hObject    handle to IntFac2 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of IntFac2 as text
%        str2double(get(hObject,'String')) returns contents of IntFac2 as a double
Val = str2double(get(handles.IntFac2,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.IntFac2,'String',1)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all

function DecFac2_Callback(~, ~, handles)
% hObject    handle to DecFac2 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of DecFac2 as text
%        str2double(get(hObject,'String')) returns contents of DecFac2 as a double

Val = str2double(get(handles.DecFac2,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.DecFac2,'String',1)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all

function SubDecFac2_Callback(~, ~, handles)
% hObject    handle to SubDecFac2 (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of SubDecFac2 as text
%        str2double(get(hObject,'String')) returns contents of SubDecFac2 as a double

Val = str2double(get(handles.SubDecFac2,'String'));

if isnan(Val) || ~isreal(Val) || Val<=0
    set(handles.SubDecFac2,'String',1)
    warndlg('Input must be numerical, real and positive!')
end
clearvars all


% --- Executes on button press in AlignPeak.
function AlignPeak_Callback(~, ~, handles)
% hObject    handle to AlignPeak (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%% Get handles
OldZLP  = get(handles.LoadPeak,'UserData');
Modi    = get(handles.SubMeth,'String');
DecModi = get(handles.DecMeth,'String');
No      = get(handles.PeakNo,'Value');

switch No
    case 1
        ZLP         = load(fullfile(tempdir,'ZLP1.mat')).ZLP;
        ZLPEnergies = load(fullfile(tempdir,'ZLP1.mat')).ZLPEnergies;
        PreZLP      = load(fullfile(tempdir,'ZLP1.mat')).PreZLP;
        ZLPDrift    = load(fullfile(tempdir,'ZLP1.mat')).ZLPDrift;
        ZLPLog      = load(fullfile(tempdir,'ZLP1Log.mat')).ZLPLog;
    case 2
        ZLP         = load(fullfile(tempdir,'ZLP2.mat')).ZLP;
        ZLPEnergies = load(fullfile(tempdir,'ZLP2.mat')).ZLPEnergies;
        PreZLP      = load(fullfile(tempdir,'ZLP2.mat')).PreZLP;
        ZLPDrift    = load(fullfile(tempdir,'ZLP2.mat')).ZLPDrift;
        ZLPLog      = load(fullfile(tempdir,'ZLP2Log.mat')).ZLPLog;
end

[AlignedZLP,ZLPVAR,AlignZLPLog] = Align3DZLP(handles,ZLP,ZLPEnergies,ZLPDrift,PreZLP,ZLPLog);
if isempty(AlignedZLP)
    clear global
    clearvars all
    return
end
%% Set handles

switch No
    case 1
        set(handles.LoadPeak,'UserData',{AlignedZLP,ZLPVAR;OldZLP{2,:}});
        if size(Modi,1)==3
            set(handles.SubMeth,'String',[Modi;'ZLP 1']);
            set(handles.SubMeth,'Value',4);
            set(handles.DecMeth,'String',[DecModi;'ZLP 1']);
            set(handles.DecMeth,'Value',5);
        elseif size(Modi,1)==4 && any(Modi{4,:}=='2')
            set(handles.SubMeth,'String',[Modi(1:3,:);'ZLP 1';'ZLP 2';'Dual ZLP.']);
            set(handles.SubMeth,'Value',6);
            set(handles.DecMeth,'String',[DecModi(1:4,:);'ZLP 1';'ZLP 2';'Dual ZLP']);
            set(handles.DecMeth,'Value',5);
        end
        
        save(fullfile(tempdir,'AlignZLP1Log.mat'),'AlignZLPLog');
    case 2
        set(handles.LoadPeak,'UserData',{OldZLP{1,:};AlignedZLP,ZLPVAR});
        if size(Modi,1)==3
            set(handles.SubMeth,'String',[Modi;'ZLP 2']);
            set(handles.SubMeth,'Value',4);
            set(handles.DecMeth,'String',[DecModi(1:4,:);'ZLP 2']);
            set(handles.DecMeth,'Value',5);
        elseif size(Modi,1)==4 && any(Modi{4,:}=='1')
            set(handles.SubMeth,'String',[Modi(1:3,:);'ZLP 1';'ZLP 2';'Dual ZLP.']);
            set(handles.SubMeth,'Value',6);
            set(handles.DecMeth,'String',[DecModi(1:4,:);'ZLP 1';'ZLP 2';'Dual ZLP']);
            set(handles.DecMeth,'Value',7);
        end
        save(fullfile(tempdir,'AlignZLP2Log.mat'),'AlignZLPLog');
end

set(handles.ShowPeak,'Visible','on');
clear global
clearvars all
% --- Executes on button press in ShowPeak.
function ShowPeak_Callback(~, ~, handles)
% hObject    handle to ShowPeak (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
ShowPeak(handles)

% --- Executes on selection change in PeakNo.
function PeakNo_Callback(~, ~, handles)
% hObject    handle to PeakNo (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: contents = cellstr(get(hObject,'String')) returns PeakNo contents as cell array
%        contents{get(hObject,'Value')} returns selected item from PeakNo

%% Get handles
No     = get(handles.PeakNo,'Value');
ZLPs   = get(handles.LoadPeak,'UserData');
        
switch No
    case 1
        ZLP = load(fullfile(tempdir,'ZLP1.mat')).ZLP;
        zlp = ZLPs{1,1};
    case 2
        ZLP = load(fullfile(tempdir,'ZLP2.mat')).ZLP;
        zlp = ZLPs{2,1};
end
if isempty(ZLP)
    set(handles.AlignPeak,'Visible','off');
    set(handles.Delete,'Visible','off');
    set(handles.ShowZLPLog,'Visible','off');
    if ~isempty(zlp)
        set(handles.ShowPeak,'Visible','on');
    else
        set(handles.ShowPeak,'Visible','off'); 
    end
else
    set(handles.AlignPeak,'Visible','on');
    set(handles.Delete,'Visible','on');
    set(handles.ShowZLPLog,'Visible','on');
    if ~isempty(zlp)
        set(handles.ShowPeak,'Visible','on');
    else
        set(handles.ShowPeak,'Visible','off');
    end
end


% --- Executes on button press in ShowZLPLog.
function ShowZLPLog_Callback(~, ~, handles)
% hObject    handle to ShowZLPLog (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
PeakNo = get(handles.PeakNo,'Value');
switch PeakNo
    case 1
        AlignZLPLog = {'ZLP 1: '};
        AlignZLPLog{end+1,1}=[];
        AlignZLPLog = [AlignZLPLog;load(fullfile(tempdir,'AlignZLP1Log.mat')).AlignZLPLog];
    case 2
        AlignZLPLog = {'ZLP 2: '};
        AlignZLPLog{end+1,1}=[];
        AlignZLPLog = [AlignZLPLog;load(fullfile(tempdir,'AlignZLP2Log.mat')).AlignZLPLog];
end
fileID = fopen(fullfile(tempdir,'LogbookZLP.txt'),'wt');
for i=1:size(AlignZLPLog,1)
    fprintf(fileID,'%s\n',AlignZLPLog{i});
end
fclose(fileID);
winopen(fullfile(tempdir,'LogbookZLP.txt'))

% --- Executes on button press in DeleteFit.
function DeleteFit_Callback(~, ~, handles)
% hObject    handle to DeleteFit (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer

if isempty(tracer)
    tracer=1;
else
    return
end
%% Set handles
set(handles.RemZLP,'Visible','off');
set(handles.RemZLP,'Value',0);
set(handles.DecIter,'Visible','off');
set(handles.DecSubIter,'Visible','off');
set(handles.text43,'Visible','off');
set(handles.DeconvSub,'Visible','off');
set(handles.SubDecFac1,'Visible','off');
set(handles.SubDecFac2,'Visible','off');
set(handles.DecMeth,'Visible','off');
set(handles.DecEELS,'Visible','off');
set(handles.DecMeth,'Value',1);
set(handles.DecFac1,'Visible','off');
set(handles.DecFac2,'Visible','off');
set(handles.DecIter,'Visible','off');
set(handles.DeleteFit,'Visible','off');
set(handles.text42,'Visible','off');

%% Save data
Data = {[],[],[],[],[]};
Str  = '';
save(fullfile(tempdir,'SubtrZLP.mat'),'Data');
save(fullfile(tempdir,'SubtrZLPLog.mat'),'Str');

UpdatePlot(handles);
clearvars all
clear global

function IntDev_Callback(~, ~, handles)
% hObject    handle to IntDev (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of IntDev as text
%        str2double(get(hObject,'String')) returns contents of IntDev as a double
Val = str2double(get(handles.IntDev,'String'));

if isnan(Val) || ~isreal(Val) || Val<0 || Val>100
    set(handles.IntDev,'String',0)
    warndlg('Input must be numerical, real, positive and between 0 and 100!')
end
clearvars all

function FineAlign_Callback(~,~,~)
% hObject    handle to FineAlign (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)


function ReadStd_Callback(~, ~, handles)
% hObject    handle to ReadStd (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of ReadStd as text
%        str2double(get(hObject,'String')) returns contents of ReadStd as a double

Val = str2double(get(handles.ReadStd,'String'));

if isnan(Val) || ~isreal(Val) || Val<0
    set(handles.ReadStd,'String',0)
    warndlg('Input must be numerical, real and not-negative!')
end
clearvars all

function ReadFull_Callback(~,~, handles)
% hObject    handle to ReadFull (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of ReadFull as text
%        str2double(get(hObject,'String')) returns contents of ReadFull as a double
Energies = get(handles.ZLPAlign,'UserData');
Val      = str2double(get(handles.ReadFull,'String'));
Val2     = str2double(get(handles.ReadStd,'String'));
Fac      = sqrt(size(Energies,2));

if isnan(Val) || ~isreal(Val) || Val<Fac*Val2
    set(handles.ReadFull,'String',Fac.*Val2)
    warndlg('Input must be numerical, real and bigger than Val of all channels combined!')
end
    
clearvars all

function BetaVal_Callback(~, ~, handles)
% hObject    handle to BetaConv (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of BetaConv as text
%        str2double(get(hObject,'String')) returns contents of BetaConv as a double

Val  = str2double(get(handles.BetaConv,'String'));
Back = get(handles.Background,'UserData');
if isnan(Val) || ~isreal(Val) || Val<=0 || Val>1
    set(handles.BetaConv,'String',1)
    warndlg('Input must be numerical, real and between 0 and 1!')
    clearvars all
    return
end

Back = Back{2};
if ~isempty(Back)
    image   = load(fullfile(tempdir,'ShowOrg.mat')).image;
    Add     = CreatePadding(handles,image);
    save(fullfile(tempdir,'Add.mat'),'Add');
end
clearvars all


% --- Executes on button press in ZLP2D.
function ZLP2D_Callback(~, ~, handles)
% hObject    handle to ZLP2D (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer

if isempty(tracer)
    tracer=1;
else
    return
end

[FileName,PathName] = uigetfile({'*.dm4'},'Select File!');
if isequal(FileName,0)
    clearvars all
    clear global
    return
end

[tags, image] = dmread(fullfile(PathName, FileName));
image         = round(double(image),8);
SpecBin       = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.hbin.Value);
VertBin       = double(tags.ImageList.Unnamed1.ImageTags.Acquisition.Parameters.Detector.vbin.Value);
SpecBin       = round(SpecBin,8);
VertBin       = round(VertBin,8);
SizeY         = size(image,1);
SizeX         = size(image,2);
FramePara     = get(handles.FramePara,'UserData');
VertBinning   = str2double(get(handles.VertBinning,'String'));
SpecBinning   = str2double(get(handles.SpecBinning,'String'));
GainRef1D     = get(handles.CorrGainRef,'UserData');
GainRef       = get(handles.Load2DRef,'UserData');
GainRef       = GainRef{2};

if VertBinning~=VertBin || SpecBinning~=SpecBin
    warndlg('Upload 2D ZLP with the same spectral and vertical binning!')
    clearvars all
    clear global
    return
end

GainRef2D     = get(handles.Load2DRef,'UserData');
NewGainRef2D  = GainRef2D{2};
GainRef2D     = GainRef2D{1};

if ~isempty(GainRef2D) && ~all(GainRef1D==1)
    GainRef2D    =    GainRef2D(FramePara(1):FramePara(2),FramePara(3):FramePara(4));
    
    if VertBinning>1
        a             = 1;
        GainRefBin    = zeros(SizeY,size(GainRef2D,2));
        NewGainRefBin = zeros(SizeY,size(NewGainRef2D,2));
        while a<= SizeY
            GainRefBin(a,:)               = sum(   GainRef2D(1:VertBinning,:),1);
            NewGainRefBin(a,:)            = sum(NewGainRef2D(1:VertBinning,:),1);
            GainRef2D(1:VertBinning,:)    = [];
            NewGainRef2D(1:VertBinning,:) = [];
            a                             = a+1;
        end
        GainRef2D    =    GainRefBin;
        NewGainRef2D = NewGainRefBin;
    end
    
    
    if SpecBinning>1
        a             = 1;
        GainRefBin    = zeros(SizeY,SizeX);
        NewGainRefBin = zeros(SizeY,SizeX);
        while a<= SizeX
            GainRefBin(:,a)               = sum(   GainRef2D(:,1:VertBinning),2);
            NewGainRefBin(:,a)            = sum(NewGainRef2D(:,1:VertBinning),2);
            GainRef2D(:,1:SpecBinning)    = [];
            NewGainRef2D(:,1:SpecBinning) = [];
            a                             = a+1;
        end
        GainRef2D    =    GainRefBin;
        NewGainRef2D = NewGainRefBin;
    end
    %% Nachbearbeiten
    image = image.*GainRef2D./NewGainRef2D;
end

if ~isempty(image)
    try
        %% Display 2D ZLP
        figure;
        imagesc(gca,image)
        title('2D ZLP')
        colormap('gray')
        axis equal
        ylim([1,SizeY])
        xlim([1,SizeX])
        
        colorbar;
        
        %% Log ZLP
        image(image<=0)=1;
        figure;
        imagesc(gca,log(image))
        title('Logarithmic 2D ZLP')
        colormap('gray')
        axis equal
        ylim([1,SizeY])
        xlim([1,SizeX])
        
        colorbar;
        
        %% Set handles
        set(handles.ZLP2D,'UserData',image);
        set(handles.CorrGainRef,'String','Replace 2D Gain Ref');
    catch
        warning('Wrong data format!');
        clearvars all
        clear global
        return
    end
    
else
    warning('Wrong data format!');
    clearvars all
    clear global
    return
end
if ~isempty(GainRef)
    set(handles.CorrSat,'Visible','on');
end
clearvars all
clear global

% --- Executes on button press in SaveNoise.
function SaveNoise_Callback(~, ~, handles)
% hObject    handle to SaveNoise (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%% Get Handles
PSF = get(handles.PSF,'UserData');
if isempty(PSF{1,1})
   msgbox('Upload EELS PSF first!')
else
   Data = SavePara(handles);
   save([pwd,'\NoiseParams.para'],'Data');
   msgbox('Noise parameters saved!')
end

% --- Executes on button press in EstimatePhi.
function EstimatePhi_Callback(~, ~, handles)
% hObject    handle to EstimatePhi (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

global tracer

if isempty(tracer)
    tracer=1;
else
    return
end

%% Get Handles
Drift     = get(handles.ShowDrift,'UserData');
EnergyRes = get(handles.Slider, 'SliderStep');
EnergyRes = EnergyRes(1);
Back      = get(handles.Background,'UserData');
%% Calculate
Drift     = round(Drift./EnergyRes);
SizeX     = size(Drift,2);
SizeY     = size(Drift,1);
Drift     = reshape(Drift,[1,SizeX*SizeY]);
Sum       = 0;
Size      = SizeX*SizeY;
n         = Size*(Size-1);

a         = 0;
WaitTics  = floor(SizeX*SizeY/500);
WaitPhi   = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);

while Size>1
    Test       = Drift(1,1);
    Drift(:,1) = [];
    Diff       = min(abs(Drift-Test),1);
    Sum        = Sum + sum(Diff,2).*2;
    Size       = size(Drift,2);
    %% Check Waitbar exist
    if ~mod(a,500)
        Flag      = CheckWaitbar1(WaitPhi);
        if Flag==1
            clearvars all
            clear global
            return
        end
        clearvars Flag
        WaitPhi.Send;
    end
    a = a+1;   
end

phi     = Sum/n;
clearvars nimg Sum
WaitPhi.Destroy;

set(handles.PhiVal,'String',round(phi,3));

Back    = Back{2};
if ~isempty(Back)
    image   = load(fullfile(tempdir,'ShowOrg.mat')).image;
    Add     = CreatePadding(handles,image);
    save(fullfile(tempdir,'Add.mat'),'Add');
end

clearvars all
clear global

% --- Executes on button press in BGArtifacts.
function BGArtifacts_Callback(~, ~, handles)
% hObject    handle to BGArtifacts (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer

if isempty(tracer)
    tracer=1;
else
    return
end
FWHM       = get(handles.ShowFWHM,'UserData');
DarkSub    = get(handles.DarkSub.SelectedObject,'String');
UserData   = get(handles.BGArtifacts,'UserData');
Dark       = UserData{1,:};
HQDark     = UserData{2,:};
Energies   = UserData{3,:};
EnergyRes  = get(handles.Slider, 'SliderStep');
EnergyRes  = EnergyRes(1);
Count      = 0;


switch DarkSub
    case 'HQDark'
        fig       = figure('Name','Select positions of "HQDark" Artifacts - select with click!');
        NewHQDark = HQDark;
        while 1
            try
                hqdark   = reshape(NewHQDark(1,1,:),[1,size(NewHQDark,3)]);
                dark     = reshape(Dark(1,1,:),[1,size(NewHQDark,3)]);
                Diff     = hqdark -dark;
                Max      = max(abs(Diff));
                set(fig,'WindowButtonMotionFcn',@(src,evt) RemArt1DDark(fig,src,evt,Energies,EnergyRes,hqdark,dark,'HQDark','Dark'));
                set(fig,'WindowButtonDownFcn',@(src,evt) Clicker(src,evt));
                plot(Energies,Diff)
                ylabel( 'Intensity [Counts]' );
                xlabel( 'Electron energy-loss [eV]' );
                xlim([min(Energies),max(Energies)]);
                ylim([-1.5*Max,1.5*Max]);
                legend('Difference HQDark - Dark')
                uiwait;
                E = get(gca,'CurrentPoint');
                E = EnergyRes*round(E(1,1)./EnergyRes);
                if isvalid(fig)
                    DelInd = zeros(1,size(E,1));
                    Count = Count + length(DelInd);
                    for i=1:length(DelInd)
                        delind        = find(Energies>=E(i),1,'first');
                        delind1       = max(1,delind-5);
                        delind2       = min(size(hqdark,2),delind+5);
                        Diff          = hqdark - dark;
                        Deriv         = Diff(1,delind1:delind2) - circshift(Diff(1,delind1:delind2),+1,2) ;
                        Deriv         = Deriv - circshift(Deriv,-1,2);
                        Deriv         = Deriv(3:end-2);
                        [~,Max]       = max(abs(Deriv),[],2);
                        DelInd        = Max + delind1+1;
                        New           = dark(1,DelInd);
                        NewHQDark(:,:,DelInd) = New;
                    end
                end
            catch
                break
            end
        end
        figure('Name','Select regions of "HQDark" Artifacts - select with click!');
        while 1
            try
                hqdark  = reshape(NewHQDark(1,1,:),[1,size(NewHQDark,3)]);
                dark    = reshape(Dark(1,1,:),[1,size(Dark,3)]);
                Diff    = hqdark-dark;
                Max     = max(abs(Diff));
                plot(Energies,Diff)
                ylabel( 'Intensity [Counts]' );
                xlabel( 'Electron energy-loss [eV]' );
                legend('Difference HQDark - Dark')
                xlim([min(Energies),max(Energies)]);
                ylim([-1.5*Max,1.5*Max]);
                Start   = 0;
                Width   = 5*mean(FWHM,'all');
                Select  = drawrectangle('Position',[Start-Width/2,-1.5*Max,Width,3*Max],'Color','b','StripeColor','w');
                l       = addlistener(Select,'ROIClicked',@(src,evt) DoubleClicker(src,evt));
                uiwait;
                pos     = Select.Position;
                Ind1    = pos(1);
                Ind2    = pos(1)+pos(3);
                Ind1    = find(Energies>=Ind1,1,'first');
                Ind2    = find(Energies<=Ind2,1,'last');

                NewHQDark(:,:,Ind1:Ind2) = Dark(:,:,Ind1:Ind2);
                Count = Count + Ind2-Ind1+1;
            catch
                delete(Select);
                delete(l);
                break
            end
        end
        if Count==0
            clearvars all
            clear global
            return
        end
        
        [image,Original] = SwitchDarkFrame(handles,HQDark,NewHQDark);
        Data = {Dark;NewHQDark;Energies};
    case 'Dark'
        fig     = figure('Name','Select positions of "Dark" Artifacts - select with click!');
        NewDark = Dark;
        while 1
            try
                hqdark   = reshape(HQDark(1,1,:),[1,size(HQDark,3)]);
                dark     = reshape(NewDark(1,1,:),[1,size(Dark,3)]);
                Diff     = dark-hqdark;
                Max      = max(abs(Diff));
                set(fig,'WindowButtonMotionFcn',@(src,evt) RemArt1DDark(fig,src,evt,Energies,EnergyRes,dark,hqdark,'Dark','HQDark'));
                set(fig,'WindowButtonDownFcn',@(src,evt) Clicker(src,evt));
                plot(Energies,Diff)
                ylabel( 'Intensity [Counts]' );
                xlabel( 'Electron energy-loss [eV]' );
                xlim([min(Energies),max(Energies)]);
                ylim([-1.5*Max,1.5*Max]);
                legend('Difference Dark - HQDark')
                uiwait;
                E = get(gca,'CurrentPoint');
                E = EnergyRes*round(E(1,1)./EnergyRes);
                if isvalid(fig)
                    DelInd = zeros(1,size(E,1));
                    Count = Count + length(DelInd);
                    for i=1:length(DelInd)
                        delind        = find(Energies>=E(i),1,'first');
                        delind1       = max(1,delind-5);
                        delind2       = min(size(dark,2),delind+5);
                        Deriv         = dark(1,delind1:delind2) - circshift(dark(1,delind1:delind2),+1,2) ;
                        Deriv         = Deriv - circshift(Deriv,-1,2);
                        Deriv         = Deriv(3:end-2);
                        [~,Max]       = max(abs(Deriv),[],2);
                        DelInd        = Max + delind1+1;
                        New           = hqdark(1,DelInd);
                        NewDark(1,1,DelInd) = New;
                    end
                end
            catch
                break
            end
        end
        figure('Name','Select regions of "Dark" Artifacts - select with click!');
        while 1
            try
                hqdark    = reshape(HQDark(1,1,:),[1,size(HQDark,3)]);
                dark      = reshape(NewDark(1,1,:),[1,size(NewDark,3)]);
                Diff      = dark- hqdark;
                Max       = max(abs(Diff));
                plot(Energies,Diff)
                ylabel( 'Intensity [Counts]' );
                xlabel( 'Electron energy-loss [eV]' );
                legend('Difference Dark - HQDark')
                xlim([min(Energies),max(Energies)]);
                ylim([-1.5*Max,1.5*Max]);
                Start     = 0;
                Width     = 5*mean(FWHM,'all');
                Select    = drawrectangle('Position',[Start-Width/2,-1.5*Max,Width,3*Max],'Color','b','StripeColor','w');
                l         = addlistener(Select,'ROIClicked',@(src,evt) DoubleClicker(src,evt));
                uiwait;
                pos     = Select.Position;
                Ind1    = pos(1);
                Ind2    = pos(1)+pos(3);
                Ind1    = find(Energies>=Ind1,1,'first');
                Ind2    = find(Energies<=Ind2,1,'last');
                NewDark(:,:,Ind1:Ind2) = HQDark(:,:,Ind1:Ind2);
                Count = Count + Ind2-Ind1+1;
            catch
                delete(Select);
                delete(l);
                break
            end
        end
        if Count==0
            clearvars all
            clear global
            return
        end
        [image,Original] = SwitchDarkFrame(handles,Dark,NewDark);
        Data             = {NewDark;HQDark;Energies};
end

%% Set handles
set(handles.BGArtifacts,'UserData',Data)
save(fullfile(tempdir,'ShowOrg.mat'),'image');
image = Original;
save(fullfile(tempdir,'Original.mat'),'image');
UpdatePlot(handles,0);

clearvars all
clear global

% --- Executes on button press in CorrGainRef.
function CorrGainRef_Callback(~, ~, handles)
% hObject    handle to CorrGainRef (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

global tracer

if isempty(tracer)
    tracer=1;
else
    return
end

String        = get(handles.CorrGainRef,'String');
Energies      = get(handles.ZLPAlign,'UserData');
UsedGainRef1D = get(handles.CorrGainRef,'UserData');
Back          = get(handles.Background,'UserData');
Back          = Back{2};
EnergyRes     = get(handles.Slider, 'SliderStep');
EnergyRes     = EnergyRes(1);
ZLP2D         = get(handles.ZLP2D,'UserData');
OrgGainRef2D  = get(handles.Load2DRef,'UserData');
OldGainRef2D  = OrgGainRef2D{2};
OrgGainRef2D  = OrgGainRef2D{1};
FramePara     = get(handles.FramePara,'UserData');
VertBinning   = str2double(get(handles.VertBinning,'String'));
SpecBinning   = str2double(get(handles.SpecBinning,'String'));
Processing    = get(handles.Processing,'String');
SizeY         = size(ZLP2D,1);
SizeX         = size(ZLP2D,2);
UserData      = get(handles.BGArtifacts,'UserData');
Dark          = UserData{1,:};
HQDark        = UserData{2,:};
BGEnergies    = UserData{3,:};
CorrSat       = get(handles.CorrSat,'UserData');
TargetInt     = get(handles.text71,'UserData');
TargetIntOld  = TargetInt{1};

switch contains(String,'Replace') || contains(String,'Apply')
    case 1
        WaitTics         = 4;
        WaitCount        = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
        %% Load New ZLP
        
        
        [FileName,PathName] = uigetfile({'*.dm4'},'Select File!');
        if isequal(FileName,0)
            clearvars all
            clear global
            return
        end
        [~, NewGainRef2D]   = dmread(fullfile(PathName, FileName));
        
        if isempty(NewGainRef2D) || size(NewGainRef2D,3)>1
            warndlg('Wrong data format!')
            clearvars all
            clear global
            return
        end
        
        NewGainRef2D = round(double(NewGainRef2D),8);
        NewGainRef2D = NewGainRef2D.^(-1);
        
        if ~isempty(CorrSat)
            answer = inputdlg('Type in target intensity of new Gain Map');
            Target = str2double(answer{1,1});
            if isnan(Target)
                warndlg('Input not numeric!')
                clearvars all
                clear global
                return
            elseif Target<1
                warndlg('Target cannot be lower than 1!')
                clearvars all
                clear global
                return
            end
            NewGainRef2D = Target.*NewGainRef2D;
            NewGainRef2D = CorrSat(NewGainRef2D).*NewGainRef2D;
            NewGainRef2D = NewGainRef2D./mean(NewGainRef2D,'all');
        end
        
        Std          = std(NewGainRef2D,1,'all');
        Mean         = mean(NewGainRef2D,'all');
        
        if size(NewGainRef2D,1)~=2048 || size(NewGainRef2D,2)~=2048
            warndlg('Please upload full 2048 x 2048 Gain Reference Map!')
            clearvars all
            clear global
            return
        end
        
        %% Plot
        figure;
        imagesc(NewGainRef2D)
        caxis([Mean-Std,Mean+Std])
        
        set(gca,'YDir','reverse')
        title('New EELS Gain Reference Map')
        axis equal
        xlim([1,size(NewGainRef2D,2)])
        ylim([1,size(NewGainRef2D,1)])
        colorbar;
        
        set(handles.GainStd,'String',num2str(round(std(NewGainRef2D,1,'all'),4)))
        %% Calculate EELS Position on the Detector and Binning
        NewGainRef2D = NewGainRef2D(FramePara(1):FramePara(2),FramePara(3):FramePara(4));
        orggainref2D = OrgGainRef2D(FramePara(1):FramePara(2),FramePara(3):FramePara(4));
        
        if VertBinning>1
            a             = 1;
            GainRefBin    = zeros(SizeY,size(orggainref2D,2));
            NewGainRefBin = zeros(SizeY,size(NewGainRef2D,2));
            while a<= SizeY
                GainRefBin(a,:)               = sum(orggainref2D(1:VertBinning,:),1);
                NewGainRefBin(a,:)            = sum(NewGainRef2D(1:VertBinning,:),1);
                orggainref2D(1:VertBinning,:) = [];
                NewGainRef2D(1:VertBinning,:) = [];
                a                             = a+1;
            end
            orggainref2D =    GainRefBin;
            NewGainRef2D = NewGainRefBin;
        end
        
        
        if SpecBinning>1
            a             = 1;
            GainRefBin    = zeros(SizeY,SizeX);
            NewGainRefBin = zeros(SizeY,SizeX);
            while a<= SizeX
                GainRefBin(:,a)               = sum(orggainref2D(:,1:VertBinning),2);
                NewGainRefBin(:,a)            = sum(NewGainRef2D(:,1:VertBinning),2);
                orggainref2D(:,1:SpecBinning) = [];
                NewGainRef2D(:,1:SpecBinning) = [];
                a                             = a+1;
            end
            orggainref2D =    GainRefBin;
            NewGainRef2D = NewGainRefBin;
        end
        
        
        figure;
        imagesc(orggainref2D)
        set(gca,'YDir','reverse')
        axis equal
        caxis([1 - Std,1 + Std])
        xlim([1,size(orggainref2D,2)])
        ylim([1,size(orggainref2D,1)])
        title('Original EELS Gain Map')
        colorbar;
        
        figure;
        imagesc(NewGainRef2D)
        set(gca,'YDir','reverse')
        axis equal
        caxis([1 - Std,1+ Std])
        xlim([1,size(NewGainRef2D,2)])
        ylim([1,size(NewGainRef2D,1)])
        title('New EELS Gain Map')
        colorbar;
        
        %% Correcting 2DZLP and Weighting Gains with it
        if ~isempty(ZLP2D)
            switch Processing
                case 'Gain Normalized'
                    ZLP2D        = ZLP2D.*orggainref2D;
            end
            
            ZLP2DNew     = ZLP2D./NewGainRef2D;
            ProbeDist    = sum(ZLP2D,2);
            ProbeDist    = ProbeDist./mean(ProbeDist);
            ProbeDist    = repmat(ProbeDist,[1,size(ZLP2D,2)]);
            
            OldGainRef1D = ProbeDist.*orggainref2D;
            OldGainRef1D = mean(OldGainRef1D,1);
            OldGainRef1D = reshape(OldGainRef1D,[1,1,size(ZLP2D,2)]);
            
            NewGainRef1D = ProbeDist.*NewGainRef2D;
            NewGainRef1D = mean(NewGainRef1D,1);
            NewGainRef1D = reshape(NewGainRef1D,[1,1,size(ZLP2D,2)]);
            String       = 'Reset 2D Gain Ref';
        else
            ZLP2DNew     = [];
            OldGainRef1D = mean(orggainref2D,1);
            NewGainRef1D = mean(NewGainRef2D,1);
            OldGainRef1D = reshape(OldGainRef1D,[1,1,size(OldGainRef1D,2)]);
            NewGainRef1D = reshape(NewGainRef1D,[1,1,size(NewGainRef1D,2)]);
            String       = 'Reset Gain Ref';
        end
        
        figure;
        plot(reshape(OldGainRef1D,[1,size(OldGainRef1D,3)]))
        switch Processing
            case 'Gain Normalized'
                hold on
                plot(reshape(NewGainRef1D,[1,size(NewGainRef1D,3)]))
                legend('Old Gain Ref','New Gain Ref')
        end
        xlabel('Pixel')
        ylabel('Gain Ref [a.u.]')
        title('1D Mean Gain Ref.')
        
        switch Processing
            case 'Gain Normalized'
                NewGainRef1D   = NewGainRef1D./OldGainRef1D;
        end
        set(handles.text71,'UserData',{TargetIntOld,TargetInt})
    case 0
        
        WaitTics        = 2;
        WaitCount       = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
        NewGainRef2D    = OrgGainRef2D;
        NewGainRef1D    = ones(size(UsedGainRef1D));
        
        %% Calculate EELS Position on the Detector and Binning
        NewGainRef2D = NewGainRef2D(FramePara(1):FramePara(2),FramePara(3):FramePara(4));
        
        if ~isempty(ZLP2D)
            if VertBinning>1
                a             = 1;
                NewGainRefBin = zeros(SizeY,size(NewGainRef2D,2));
                
                while a<= SizeY
                    NewGainRefBin(a,:)            = sum(NewGainRef2D(1:VertBinning,:),1);
                    NewGainRef2D(1:VertBinning,:) = [];
                    a                             = a+1;
                end
                NewGainRef2D = NewGainRefBin;
            end
            
            
            if SpecBinning>1
                a             = 1;
                NewGainRefBin = zeros(SizeY,SizeX);
                while a<= SizeX
                    NewGainRefBin(:,a)            = sum(NewGainRef2D(:,1:VertBinning),2);
                    NewGainRef2D(:,1:SpecBinning) = [];
                    a                             = a+1;
                end
                NewGainRef2D = NewGainRefBin;
            end
            
            %% Undo Gain correction on 2DZLP
            
            ZLP2DNew    = ZLP2D.*OldGainRef2D./NewGainRef2D;
            switch Processing
                case 'Gain Normalized'
                    String      = 'Replace 2D Gain Ref';
                otherwise
                    String      = 'Apply 2D Gain Ref';
            end
        else
            ZLP2DNew    = [];
            switch Processing
                case 'Gain Normalized'
                    String      = 'Replace Gain Ref';
                otherwise
                    String      = 'Apply Gain Ref';
            end
        end
        set(handles.text71,'UserData',{TargetIntOld,TargetIntOld})
end
[image,Original] = SwitchGainRef(handles,UsedGainRef1D,NewGainRef1D);
Norm             = sum(image,3);
NewDark          =   Dark.*UsedGainRef1D./NewGainRef1D;
NewHQDark        = HQDark.*UsedGainRef1D./NewGainRef1D;
Data             = {NewDark;NewHQDark;BGEnergies};

WaitCount.Send;
%% Update FWHM Map
maxEne    = max(image,[],3);
minEne    = min(image,[],3);
ZLP       =(minEne + maxEne) / 2;
SizeX     = size(image,2);
SizeY     = size(image,1);
SizeZ     = size(image,3);
FWHM      = zeros([SizeY,SizeX]);
Prec      = -round(log10(EnergyRes));

parfor i=1:SizeY
    for j=1:SizeX
        img             = reshape(image(i,j,:),[1,1,SizeZ]);
        % Find where the data first drops below half the max.
        index1          = find(img >= ZLP(i,j), 1, 'first');
        % Find where the data last rises above half the max.
        index2          = find(img >= ZLP(i,j), 1, 'last');
        if index1~=index2
            FWHM(i,j)       = Energies(1,index2)-Energies(1,index1);
        else
            FWHM(i,j)       = NaN;
        end
    end
end
FWHM(isnan(FWHM)) = round(mean(FWHM(~isnan(FWHM)),'all'),Prec);

%% Set Handles
set(handles.GainStd,'String',num2str(round(std(NewGainRef2D.^(-1),1,'all'),4)));
set(handles.ShowFWHM,'UserData',FWHM);
set(handles.ShowInt,'UserData',Norm);
set(handles.CorrGainRef,'UserData',NewGainRef1D);
set(handles.ZLP2D,'UserData',ZLP2DNew);
set(handles.Load2DRef,'UserData',{OrgGainRef2D,NewGainRef2D})
set(handles.CorrGainRef,'String',String)
set(handles.BGArtifacts,'UserData',Data);
%% Save files
save(fullfile(tempdir,'ShowOrg.mat'),'image');

if ~isempty(Back)
    Add = CreatePadding(handles,image);
    save(fullfile(tempdir,'Add.mat'),'Add');
end

image = Original;
save(fullfile(tempdir,'Original.mat'),'image');
WaitCount.Send
WaitCount.Destroy

%% Update Plot
UpdatePlot(handles,0);

clearvars all
clear global

% --- Executes when selected object is changed in DarkSub.
function DarkSub_SelectionChangedFcn(~, ~, handles)
% hObject    handle to the selected object in DarkSub
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer

if isempty(tracer)
    tracer=1;
else
    return
end

DarkSub    = get(handles.DarkSub.SelectedObject,'String');
DarkSubOld = get(handles.DarkSub,'UserData');
UserData   = get(handles.BGArtifacts,'UserData');
Dark       = UserData{1};
HQDark     = UserData{2};
GainRef    = get(handles.CorrGainRef,'UserData');
Back       = get(handles.Background,'UserData');
Back       = Back{2};

%% Correct Original Image
switch DarkSubOld
    case 1
        OldFrame = HQDark./GainRef;
    case 2
        OldFrame = Dark./GainRef;
    case 3
        OldFrame = [];
end

switch DarkSub
    case 'HQDark'
        NewFrame = HQDark./GainRef;
    case 'Dark'
        NewFrame = Dark./GainRef;
    otherwise
        NewFrame = [];
end

[image,Original] = SwitchDarkFrame(handles,OldFrame,NewFrame);
if isempty(image)
    clearvars all
    clear global
end

switch DarkSub
    case 'HQDark'
        set(handles.BGArtifacts,'Visible','on');
        set(handles.DarkSub,'UserData',1)
    case 'Dark'
        set(handles.BGArtifacts,'Visible','on');
        set(handles.DarkSub,'UserData',2)
    otherwise
        set(handles.BGArtifacts,'Visible','off');
        set(handles.DarkSub,'UserData',3)
end

%% Set handles
if ~isempty(Back)
    Add = CreatePadding(handles,image);
    save(fullfile(tempdir,'Add.mat'),'Add');
end
save(fullfile(tempdir,'ShowOrg.mat'),'image');
image = Original;
save(fullfile(tempdir,'Original.mat'),'image');

UpdatePlot(handles,0);
clearvars all
clear global

% --- Executes on button press in Load2DRef.
function Load2DRef_Callback(~, ~, handles)
% hObject    handle to Load2DRef (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer

if isempty(tracer)
    tracer=1;
else
    return
end

[FileName,PathName] = uigetfile({'*.dm4'},'Select File!');
if isequal(FileName,0)
    clearvars all
    clear global
    return
end
[~, GainRef]   = dmread(fullfile(PathName, FileName));

if isempty(GainRef) || size(GainRef,3)>1
    warndlg('Wrong data format!')
    clearvars all
    clear global
    return
end
ZLP2D        = get(handles.ZLP2D,'UserData');
GainRef      = round(double(GainRef),8);
GainRef      = GainRef.^(-1);
Std          = std(GainRef,1,'all');
Mean         = mean(GainRef,'all');
CorrSat      = get(handles.CorrSat,'UserData');

if size(GainRef,1)~=2048 || size(GainRef,2)~=2048
    warndlg('Please upload full 2048 x 2048 Gain Reference Map!')
    clearvars all
    clear global
    return
end

FramePara    = get(handles.FramePara,'UserData');
VertBinning  = str2double(get(handles.VertBinning,'String'));
SpecBinning  = str2double(get(handles.SpecBinning,'String'));
String       = get(handles.CorrGainRef,'String');

if ~(contains(String,'Replace') || contains(String,'Apply'))
    clear global
    CorrGainRef_Callback(1, 1, handles);
    
    global tracer
    tracer = 1;
end

answer = inputdlg('Type in target intensity of new Gain Map');
Target = str2double(answer{1,1});
if isnan(Target)
    warndlg('Input not numeric!')
    clearvars all
    clear global
    return
elseif Target<1
    warndlg('Target cannot be lower than 1!')
    clearvars all
    clear global
    return
end


if ~isempty(CorrSat)
    CorrSat_Callback(1, 1, handles)
end

%% Crop and bin for EELS Gain Ref

EELSRef = GainRef(FramePara(1):FramePara(2),FramePara(3):FramePara(4));
if VertBinning>1
    a             = 1;
    EELSRefBin    = zeros(SizeY,size(EELSRef2D,2));
    while a<= SizeY
        EELSRefBin(a,:)             = sum(EELSRef(1:VertBinning,:),1);
        EELSRef(1:VertBinning,:)    = [];
        a                           = a+1;
    end
    EELSRef = EELSRefBin;
end


if SpecBinning>1
    a             = 1;
    EELSRefBin    = zeros(SizeY,SizeX);
    while a<= SizeX
        EELSRefBin(:,a)             = sum(EELSRef(:,1:VertBinning),2);
        EELSRef(:,1:SpecBinning)    = [];
        a                           = a+1;
    end
    EELSRef = EELSRefBin;
end

%% Plot
figure;
imagesc(EELSRef)
caxis([Mean-Std,Mean+Std])

set(gca,'YDir','reverse')
title('EELS Gain Reference Map')
axis equal
xlim([1,size(EELSRef,2)])
ylim([1,size(EELSRef,1)])
colorbar;

figure;
imagesc(GainRef)
caxis([Mean-Std,Mean+Std])

set(gca,'YDir','reverse')
title('Full Gain Reference Map')
axis equal
xlim([1,size(GainRef,2)])
ylim([1,size(GainRef,1)])
colorbar;


%% Set handles
set(handles.Load2DRef,'UserData',{GainRef,EELSRef});
set(handles.text71,'UserData',{Target,Target});
set(handles.CorrGainRef,'Visible','on');

if ~isempty(ZLP2D)
    set(handles.CorrSat,'Visible','on');
end
clearvars all
clear global

function GainStd_Callback(~, ~, handles)
% hObject    handle to GainStd (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

% Hints: get(hObject,'String') returns contents of GainStd as text
%        str2double(get(hObject,'String')) returns contents of GainStd as a double

Val = str2double(get(handles.GainStd,'String'));

if isnan(Val) || ~isreal(Val) || Val<0
    set(handles.GainStd,'String',0)
    warndlg('Input must be numerical, real and not-negative!')
end
clearvars all

% --- Executes on button press in CorrSat.
function CorrSat_Callback(~, ~, handles)
% hObject    handle to CorrSat (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)
global tracer

if isempty(tracer)
    tracer=1;
else
    return
end

ZLP2D               = get(handles.ZLP2D,'UserData');
GainRef             = get(handles.Load2DRef,'UserData');
OldGainRef          = GainRef{1};
GainRef             = GainRef{2};
String              = get(handles.CorrSat,'String');
Cut                 = get(handles.EstimatePhi,'UserData');
if ~isempty(Cut)
    Cut1                = Cut(1);
    Cut2                = Cut(2);
end
Drift               = get(handles.ShowDrift,'UserData');
EnergyRes           = get(handles.Slider, 'SliderStep');
EnergyRes           = EnergyRes(1);
image               = load(fullfile(tempdir,'ShowOrg.mat')).image;
Original            = load(fullfile(tempdir,'Original.mat')).image;
Drift               = round(Drift./EnergyRes);
TargetInt           = get(handles.text71,'UserData');

if contains(String,'Correct')
    [fileName,pathName] = uigetfile('*.mat');
    
    if isequal(pathName,0)
        clearvars all
        clear global
        return
    end
    
    try
        CorrSat             = load(fullfile(pathName,fileName)).Corr;
    catch
        warndlg('Wrong file selected!')
        clearvars all
        clear global
        return
    end
    Target = TargetInt{2};
    
    GainRefCorr = GainRef.*Target;
    GainRefCorr = CorrSat(round(GainRefCorr+1)).*GainRefCorr;
    GainRefCorr = GainRefCorr./mean(GainRefCorr,'all');
    
    Target         = TargetInt{1};
    OldGainRefCorr = OldGainRef.*Target;
    OldGainRefCorr = CorrSat(round(OldGainRefCorr+1)).*OldGainRefCorr;
    OldGainRefCorr = OldGainRefCorr./mean(OldGainRefCorr,'all');
    
    zlp2DNew            = max(round(ZLP2D.*GainRef)+1,1);
    corr                = CorrSat(zlp2DNew);
    ZLP2DNew            = ZLP2D.*corr.*GainRef./GainRefCorr;
    Sum                 = sum(ZLP2D,2);
    dist                = Sum./sum(Sum,'all');
    
    Dist                = repmat(dist,[1,size(Original,3)]);
    WaitTics            = size(Original,1);
    Wait                = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
    
    for i=1:size(Original,1)
        for j =1:size(Original,2)
            spec            = reshape(Original(i,j,:),[1,size(Original,3)]);
            Img             = repmat(spec,[size(Dist,1),1]).*Dist;
            img             = round(  max(Img.*GainRef ,1)  );
            corr            = CorrSat(round(img));
            Original(i,j,:) = round(sum(Img.*corr.*GainRef./GainRefCorr,1));  
        end
        Wait.Send
    end
    Wait.Destroy;
    
    if ~isempty(Drift)
        Dist       = repmat(dist,[1,size(image,3)]);
        WaitTics    = size(image,1)*size(image,2);
        Wait        = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
        for i=1:size(image,1)
            for j =1:size(image,2)
                spec            = reshape(image(i,j,:),[1,size(image,3)]);
                Img             = repmat(spec,[size(Dist,1),1]).*Dist;
                GR              = circshift(GainRef,-Drift(i,j),2);
                GR              = GR(:,1+Cut1:end-Cut2);
                img             = round(  max(Img.* GR ,1)  );
                corr            = CorrSat(round(img));
                image(i,j,:)    = round(sum(Img.*corr.*GainRef./GainRefCorr,1));
            end
            Wait.Send
        end
        Wait.Destroy;
    else
        image    = Original;
    end
    save(fullfile(tempdir,'ShowOrg.mat'),'image');
    image = Original;
    save(fullfile(tempdir,'Original.mat'),'image');
    
    
    set(handles.CorrSat,'UserData',CorrSat);
    set(handles.CorrSat,'String','Redo Saturation')
    
else
    CorrSat             = get(handles.CorrSat,'UserData');
    Linear              = linspace(0,2^16-1,2^16);
    Assignment          = round(CorrSat(Linear+1).*Linear);
    Linear              = linspace(0,max(Assignment),max(Assignment)+1);
    Undo                = zeros(size(Linear));
    for i=1:size(Linear,2)
        Undo(1,i)       = find(Assignment<=Linear(1,i),1,'last');
    end
    CorrSatOld          = CorrSat;
    CorrSat             = Undo./(Linear+1);
    Sum                 = sum(ZLP2D,2);
    dist                = Sum./sum(Sum,'all');
    
    
    Target         = TargetInt{2};
    GainRefCorr    = GainRef.*CorrSatOld(Target).*Target;
    GainRefCorr    = CorrSat(round(GainRefCorr+1)).*GainRefCorr;
    GainRefCorr    = GainRefCorr./mean(GainRefCorr,'all');
    
    Target         = TargetInt{1};
    OldGainRefCorr = OldGainRef.*CorrSatOld(Target).*Target;
    OldGainRefCorr = CorrSat(round(OldGainRefCorr+1)).*OldGainRefCorr;
    OldGainRefCorr = OldGainRefCorr./mean(OldGainRefCorr,'all');
    
    Dist                = repmat(dist,[1,size(Original,3)]);
    WaitTics            = size(Original,1);
    Wait                = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
    
    for i=1:size(Original,1)
        for j =1:size(Original,2)
            spec            = reshape(Original(i,j,:),[1,size(Original,3)]);
            Img             = repmat(spec,[size(Dist,1),1]).*Dist;
            img             = round(  max(Img.*GainRef ,1)  );
            corr            = CorrSat(round(img));
            Original(i,j,:) = round(sum(Img.*corr.*GainRef./GainRefCorr,1));
            
        end
        Wait.Send;
    end
    Wait.Destroy;
    
    if ~isempty(Drift)
        Dist       = repmat(dist,[1,size(image,3)]);
        WaitTics    = size(image,1)*size(image,2);
        Wait        = parfor_wait(WaitTics, 'Waitbar', true,'ReportInterval',1);
        for i=1:size(image,1)
            for j =1:size(image,2)
                spec            = reshape(image(i,j,:),[1,size(image,3)]);
                Img             = repmat(spec,[size(Dist,1),1]).*Dist;
                GR              = circshift(GainRef,-Drift(i,j),2);
                GR              = GR(:,1+Cut1:end-Cut2);
                img             = round(  max(Img.* GR ,1)  );
                corr            = CorrSat(round(img));
                image(i,j,:)    = round(sum(Img.*corr.*GainRef./GainRefCorr,1));
                
            end
            Wait.Send;
        end
        Wait.Destroy;
    else
        image    = Original;
    end
    
    save(fullfile(tempdir,'ShowOrg.mat'),'image');
    image = Original;
    save(fullfile(tempdir,'Original.mat'),'image');
    
    zlp2DNew            = max(round(ZLP2D)+1,1);
    corr                = CorrSat(zlp2DNew.*GainRef);
    ZLP2DNew            = ZLP2D.*corr.*GainRef./GainRefCorr;
    set(handles.CorrSat,'String','Correct Saturation')
end

set(handles.ZLP2D,'UserData',ZLP2DNew);
set(handles.Load2DRef,'UserData',{OldGainRefCorr,GainRefCorr});

clear global
UpdatePlot(handles);

clearvars all


% --- Executes on button press in PSF.
function PSF_Callback(~, ~, handles)
% hObject    handle to PSF (see GCBO)
% eventdata  reserved - to be defined in a future version of MATLAB
% handles    structure with handles and user data (see GUIDATA)

%% Load Data
[fileName,pathName] = uigetfile('*.mat');

if isequal(pathName,0)
    clearvars all
    clear global
    return
end

try
    PearsonData   = load(fullfile(pathName,fileName));
    PSFEELS       = PearsonData.KernelEELS;
    PearsonEELS   = PearsonData.PearsonEELS;
    clearvars PeasonData
catch
    warndlg('Wrong file selected!')
    clearvars all
    clear global
    return
end

%% Get Handles
Back = get(handles.Background,'UserData');

%% Calculate
N               = (size(PearsonEELS,2)+1)/2;
x               = linspace(1,N,N);
xinv            = fliplr(x);
X               = [x,xinv(1,2:end)];
BetaCorr        = N./(N-1).*(1 -  sum(X.* PearsonEELS,'all')./N.^2) ;
BetaConv        = sum(PearsonEELS./max(PearsonEELS,[],'all')).^(-1);


PearsInt        = zeros(size(PearsonEELS));
PearsInt(1,1:2) = [2/3,1/6];
PearsInt(1,end) = 1/6;
PearsInt        = PearsInt./0.625;

PInt            = ifft(fft(PearsonEELS,[],2).* fft(PearsInt,[],2),[],2);
PInt            = abs(PInt).*real(PInt);
PInt            = PInt./max(PInt,[],2);

BetaIntCorr     = N./(N-1).*(1 -  sum(X.* PInt,'all')./N.^2) ;
BetaIntConv     = sum(PInt./max(PInt,[],'all')).^(-1);

P               = sqrt(abs(fft(PearsInt,[],2)));
PearsInt        = ifft(P,[],2);
PearsInt(PearsInt<0)=0;
PearsInt        = PearsInt./sum(PearsInt);
PSFEELSInt      = ifft(fft(ifftshift(PSFEELS),[],2).*P,[],2);
PSFEELSInt      = PSFEELSInt./sum(PSFEELSInt,2);
PSFEELSInt      = fftshift(PSFEELSInt); 
PearsInt        = fftshift(PearsInt);


%% Set Handles
set(handles.BetaConv,'String',num2str(round(BetaConv,3)));
set(handles.BetaCorr,'String',num2str(round(BetaCorr,3)));
set(handles.PSF,'UserData',{PSFEELS,PSFEELSInt,BetaIntConv,BetaIntCorr,PearsInt});
set(handles.PSF,'String','Reload EELS PSF');

Back    = Back{2};
if ~isempty(Back)
    image   = load(fullfile(tempdir,'ShowOrg.mat')).image;
    Add     = CreatePadding(handles,image);
    save(fullfile(tempdir,'Add.mat'),'Add');
end


clearvars all
